/*
 * unity-webapps-binding.c
 * Copyright (C) Canonical LTD 2012
 * 
 * Author: Alexandre Abreu <alexandre.abreu@canonical.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
#include <glib.h>

#include <string.h>

#include <unity-webapps-service.h>
#include <unity-webapps-context.h>
#include <unity-webapps-music-player-context.h>
#include <unity-webapps-permissions.h>
#include <unity-webapps-launcher-context.h>
#include <unity-webapps-notification-context.h>

#include "npapi-headers/headers/npapi.h"
#include "npapi-headers/headers/npfunctions.h"
#include "npapi-headers/headers/npruntime.h"


#include "wrapped-ptr-type.h"
#include "wrapped-callback-type.h"
#include "unity-webapps-binding-test.h"
#include "unity-npapi-binding-utils.h"
#include "unity-webapps-scriptable-object.h"
#include "unity-npapi-plugin.h"
#include "unity-npapi-tools.h"

NPVariant
unity_webapps_binding_service_new (NPP instance
				   , NPObject * npobject
				   , const NPVariant *args
				   , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  UnityWebappsService * service = unity_webapps_service_new ();

  NPObject * object = create_wrapped_ptr_object_for (instance, service);
  if (NULL == object)
    {
      NPN_SetException(object, "Unable to get a new service object");
      return result;
    }

  OBJECT_TO_NPVARIANT(object, result);

  return result;
}

/**
 * 
 * @
 */
NPVariant
unity_webapps_binding_context_new_sync (NPP instance
					, NPObject * npobject
					, const NPVariant *args
					, uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 5)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer
  if (   ! NPVARIANT_IS_OBJECT(args[0])
	 || ! NPVARIANT_IS_STRING(args[1])
	 || ! NPVARIANT_IS_STRING(args[2])
	 || ( ! NPVARIANT_IS_STRING(args[3]) && ! is_null_or_void (args[3]))
	 || ( ! NPVARIANT_IS_STRING(args[4]) && ! is_null_or_void (args[4]))
	 )
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: context_new_sync");
      return result;
    }

  UnityWebappsService * service =
    (UnityWebappsService *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;
	
  const gchar * name = create_safe_string_for (&args[1]);
  const gchar * domain = create_safe_string_for (&args[2]);
  const gchar * icon_url = NULL;
  if ( ! is_null_or_void (args[3]))
    {
      icon_url = create_safe_string_for (&args[3]);
    }
  const gchar * mime_types = NULL;
  if ( ! is_null_or_void (args[4]))
    {
      mime_types = create_safe_string_for (&args[4]);
    }
  
  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  UnityWebappsContext
    * context = unity_webapps_context_new_sync (service, name, domain, icon_url, mime_types);

  g_free ((gpointer) name);
  g_free ((gpointer) domain);
  g_free ((gpointer) icon_url);
  g_free ((gpointer) mime_types);

  NPObject * object = create_wrapped_ptr_object_for (instance, context);
  if (NULL == object)
    {
      NPN_SetException(npobject, "Unable to properly wrap the returned object");
      return result;
    }
  
  OBJECT_TO_NPVARIANT(object, result);

  return result;
}


NPVariant
unity_webapps_binding_context_new_lazy (NPP instance
					, NPObject * npobject
					, const NPVariant *args
					, uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 5)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }
  
  // TODO must add proper type validation layer
  if (   ! NPVARIANT_IS_OBJECT(args[0])
	 || ! NPVARIANT_IS_STRING(args[1])
	 || ! NPVARIANT_IS_STRING(args[2])
	 || ( ! NPVARIANT_IS_STRING(args[3]) && ! is_null_or_void (args[3]))
	 || ( ! NPVARIANT_IS_STRING(args[4]) && ! is_null_or_void (args[4]))
	 )
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: context_new_lazy");
      return result;
    }

  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  UnityWebappsService *
    service = (UnityWebappsService *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;

  const gchar * name = create_safe_string_for (&args[1]);
  const gchar * domain = create_safe_string_for (&args[2]);
  const gchar * icon_url = NULL;
  if ( ! is_null_or_void (args[3]))
    {
      icon_url = create_safe_string_for (&args[3]);
    }
  const gchar * mime_types = NULL;
  if ( ! is_null_or_void (args[4]))
    {
      mime_types = create_safe_string_for (&args[4]);
    }

  UnityWebappsContext
    * context = unity_webapps_context_new_lazy (service, name, domain, icon_url, mime_types);

  g_free ((gpointer) name);
  g_free ((gpointer) domain);
  g_free ((gpointer) icon_url);
  g_free ((gpointer) mime_types);

  NPObject * object = create_wrapped_ptr_object_for (instance, context);
  if (NULL == object)
    {
      NPN_SetException(npobject, "Unable to wrap context in proper object");
      return result;
    }
  
  OBJECT_TO_NPVARIANT(object, result);

  return result;
}


static void
UnityWebappsContextPrepareCallback_dispatcher (UnityWebappsContext * context,
					       gpointer user_data)
{
  // not really safe ... 
  wrapped_callback_t * pCallbackObject = (wrapped_callback_t *) user_data;

  // fill out the arguments
  NPVariant args [2];

  // TODO call w/ meaningful values
  NULL_TO_NPVARIANT (args[0]); //context
  NULL_TO_NPVARIANT (args[1]); //user data

  NPVariant response;

  NPN_InvokeDefault (pCallbackObject->instance,
		     pCallbackObject->wrapped_callback,
		     args,
		     G_N_ELEMENTS(args),
		     &response);

  NPN_ReleaseVariantValue(&response);
}


NPVariant
unity_webapps_binding_context_prepare (NPP instance
					, NPObject * object
					, const NPVariant *args
					, uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 3)
    {
      NPN_SetException(object, "Invalid argument count in NPAPI function call");
      return result;
    }

  if (   ! NPVARIANT_IS_OBJECT(args[0])
	 || ! NPVARIANT_IS_OBJECT(args[1])
	 || ( ! NPVARIANT_IS_OBJECT(args[2]) && ! is_null_or_void (args[2]))
	 )
    {
      NPN_SetException(object, "Invalid argument type in NPAPI function call: context_prepare");
      return result;
    }
  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;

  NPObject * callback = NPVARIANT_TO_OBJECT(args[1]);
  ADD_NPOBJECT_RETAIN_FOR_CONTEXT(instance, context, callback);

  NPObject * wrappedCallback =
    create_wrapped_callback_object_for (instance, callback);
  ADD_NPOBJECT_RETAIN_FOR_CONTEXT(instance, context, wrappedCallback);

  unity_webapps_context_prepare (context
                                 , UnityWebappsContextPrepareCallback_dispatcher
                                 , wrappedCallback);

  return result;
}


NPVariant
unity_webapps_binding_context_destroy (NPP instance
				       , NPObject * npobject
				       , const NPVariant *args
				       , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 2)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer
  if ( ! NPVARIANT_IS_OBJECT(args[0])
       || ! is_boolean_convertible_value(args[1])
       )
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: context_destroy");
      return result;
    }

  UnityWebappsContext *
    context = (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;

  gboolean user_abandoned = extract_boolean_value_from(args[1]);

  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  unity_webapps_context_destroy (context, user_abandoned);

  return result;
}

NPVariant
unity_webapps_binding_context_add_icon (NPP instance
					, NPObject * npobject, const NPVariant *args
					, uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 3)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer
  if (   ! NPVARIANT_IS_OBJECT(args[0])
	 || ( ! NPVARIANT_IS_STRING(args[1]) && ! is_null_or_void(args[1]) )
         || ( ! NPVARIANT_IS_INT32(args[2]) && ! NPVARIANT_IS_DOUBLE(args[2]) )
         )
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: context_add_icon");
      return result;
    }

  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;

  const gchar * url = create_safe_string_for (&args[1]);
  gint size = 0;
  if (NPVARIANT_IS_INT32(args[2]))
    {
      size = NPVARIANT_TO_INT32(args[2]);
    }
  else
    {
      size = NPVARIANT_TO_DOUBLE(args[2]);
    }

  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  unity_webapps_context_add_icon (context, url, size);

  g_free ((gpointer) url);

  return result;
}


NPVariant
unity_webapps_binding_context_set_view_is_active (NPP instance
						  , NPObject * npobject
						  , const NPVariant *args
						  , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 2)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }
  
  // TODO must add proper type validation layer
  if (   ! NPVARIANT_IS_OBJECT(args[0])
	 || ! is_boolean_convertible_value(args[1])
	 )
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: context_set_view_is_active");
      return result;
    }
  
  UnityWebappsContext *
    context = (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;
	
  gboolean active = extract_boolean_value_from(args[1]);

  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  unity_webapps_context_set_view_is_active (context, active);
  
  return result;
}

NPVariant
unity_webapps_binding_context_set_view_location (NPP instance
						 , NPObject * npobject
						 , const NPVariant *args
						 , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 2)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer
  if (   ! NPVARIANT_IS_OBJECT(args[0])
	 || ! NPVARIANT_IS_STRING(args[1]))
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: context_set_view_location");
      return result;
    }

  UnityWebappsContext * context = (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;
	
  const gchar * location = create_safe_string_for (&args[1]);
	
  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  unity_webapps_context_set_view_location (context, location);

  g_free ((gpointer) location);

  return result;
}


static void
UnityWebappsContextRaiseCallback_dispatcher (UnityWebappsContext *context
                                             , const gchar *file
					     , gpointer user_data)
{
  // not really safe ... 
  wrapped_callback_t * pCallbackObject = (wrapped_callback_t *) user_data;

  // fill out the arguments
  NPVariant args [3];

  // TODO call w/ meaningful values
  NULL_TO_NPVARIANT (args[0]); //context 
  NULL_TO_NPVARIANT (args[1]); //file
  NULL_TO_NPVARIANT (args[2]); //user data

  if (NULL != file && 0 != strlen(file))
    {
      gint size = strlen(file) + 1;
      gchar * ret = NPN_MemAlloc (size);
      if (NULL != ret)
        {
          memset (ret, 0, size);
          strncpy (ret, file, size);
          STRINGZ_TO_NPVARIANT(ret, args[1]);
        }
    }

  NPVariant response;
  NPN_InvokeDefault (pCallbackObject->instance
		     , pCallbackObject->wrapped_callback
		     , args
		     , G_N_ELEMENTS(args)
		     , &response);
  NPN_ReleaseVariantValue(&response);
}


static void
UnityWebappsContextCloseCallback_dispatcher (UnityWebappsContext *context
					     , gpointer user_data)
{
  // not really safe ... 
  wrapped_callback_t * pCallbackObject = (wrapped_callback_t *) user_data;

  // fill out the arguments
  NPVariant args [2];

  // TODO call w/ meaningful values
  NULL_TO_NPVARIANT (args[0]); //context 
  NULL_TO_NPVARIANT (args[1]); //user data

  NPVariant response;

  NPN_InvokeDefault (pCallbackObject->instance
		     , pCallbackObject->wrapped_callback
		     , args
		     , G_N_ELEMENTS(args)
		     , &response);

  NPN_ReleaseVariantValue(&response);
}

NPVariant
unity_webapps_binding_context_on_raise_callback (NPP instance
						 , NPObject * npobject
						 , const NPVariant *args
						 , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 3)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer
  if (   ! NPVARIANT_IS_OBJECT(args[0])
	 || ! NPVARIANT_IS_OBJECT(args[1])
	 || ( ! NPVARIANT_IS_OBJECT(args[2]) && ! is_null_or_void (args[2]))
	 )
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: context_on_raise_callback");
      return result;
    }

  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;

  NPObject * callback = NPVARIANT_TO_OBJECT(args[1]);
  ADD_NPOBJECT_RETAIN_FOR_CONTEXT(instance, context, callback);

  // don't really care about the last param (simplification)

  NPObject * wrappedCallback =
    create_wrapped_callback_object_for (instance, callback);
  ADD_NPOBJECT_RETAIN_FOR_CONTEXT(instance, context, wrappedCallback);

  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  // TODO !!!!
  // mmmh make sure that it works fine (that the same dispatcher function can be registered multiple times)
  unity_webapps_context_on_raise_callback (context
					   , UnityWebappsContextRaiseCallback_dispatcher
					   , wrappedCallback);

  return result;
}


NPVariant
unity_webapps_binding_context_on_close_callback (NPP instance
						 , NPObject * npobject
						 , const NPVariant *args
						 , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 3)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer
  if (   ! NPVARIANT_IS_OBJECT(args[0])
	 || ! NPVARIANT_IS_OBJECT(args[1])
	 || ( ! NPVARIANT_IS_OBJECT(args[2]) && ! is_null_or_void (args[2]))
	 )
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: context_on_close_callback");
      return result;
    }

  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;
	
  NPObject * callback = NPVARIANT_TO_OBJECT(args[1]);
  ADD_NPOBJECT_RETAIN_FOR_CONTEXT(instance, context, callback);
	
  // don't really care about the last param (simplification)
	
  NPObject * wrappedCallback =
    create_wrapped_callback_object_for (instance, callback);
  ADD_NPOBJECT_RETAIN_FOR_CONTEXT(instance, context, wrappedCallback);

  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  // TODO !!!!
  // mmmh make sure that it works fine (that the same dispatcher function can be registered multiple times)
  unity_webapps_context_on_close_callback (context
					   , UnityWebappsContextCloseCallback_dispatcher
					   , wrappedCallback);

  return result;
}


static const gchar *
UnityWebappsContextPreviewCallback_dispatcher (UnityWebappsContext * context
					       , gpointer user_data)
{
  // not really safe ... 
  wrapped_callback_t * pCallbackObject = (wrapped_callback_t *) user_data;

  // fill out the arguments
  NPVariant args [2];
  
  // TODO call w/ meaningful values
  NULL_TO_NPVARIANT (args[0]); //context
  NULL_TO_NPVARIANT (args[1]); //user data
  
  NPVariant response;
  
  NPN_InvokeDefault (pCallbackObject->instance
		     , pCallbackObject->wrapped_callback
		     , args
		     , G_N_ELEMENTS(args)
		     , &response);
  
  // we must have received something
  if ( ! NPVARIANT_IS_STRING (response))
    {
      return NULL;
    }
  
  // TODO
  // NPN_ReleaseVariantValue(&response);
  return create_safe_string_for (&response);
}


NPVariant
unity_webapps_binding_context_set_preview_requested_callback (NPP instance
							      , NPObject * npobject
							      , const NPVariant *args
							      , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 3)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer && handle the NULL case
  if (   ! NPVARIANT_IS_OBJECT(args[0])
	 || ! NPVARIANT_IS_OBJECT(args[1])
	 || ( ! NPVARIANT_IS_OBJECT(args[2]) && ! is_null_or_void (args[2]))
	 )
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: context_set_preview_requested_callback");
      return result;
    }

  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;
 
  NPObject * callback = NPVARIANT_TO_OBJECT(args[1]);
  ADD_NPOBJECT_RETAIN_FOR_CONTEXT(instance, context, callback);

  // don't really care about the last param (simplification)

  NPObject * wrappedCallback =
    create_wrapped_callback_object_for (instance, callback);
  ADD_NPOBJECT_RETAIN_FOR_CONTEXT(instance, context, wrappedCallback);

  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  // TODO !!!!
  // mmmh make sure that it works fine (that the same dispatcher function can be registered multiple times)
  unity_webapps_context_set_preview_requested_callback (context
                                                        , UnityWebappsContextPreviewCallback_dispatcher
                                                        , wrappedCallback);

  return result;
}


void
UnityWebappsContextActionCallback_dispatcher (UnityWebappsContext *context
					      , gpointer user_data)
{
  // not really safe ... 
  wrapped_callback_t * pCallbackObject = (wrapped_callback_t *) user_data;
	
  // fill out the arguments
  NPVariant args [2];
	
  // TODO call w/ meaningful values
  NULL_TO_NPVARIANT (args[0]);
  NULL_TO_NPVARIANT (args[1]);
	
  NPVariant response;
	
  NPN_InvokeDefault (pCallbackObject->instance
                     , pCallbackObject->wrapped_callback
                     , args
                     , G_N_ELEMENTS(args)
                     , &response);

  NPN_ReleaseVariantValue (&response);
}


NPVariant
unity_webapps_binding_context_remove_application_action (NPP instance
							 , NPObject * npobject
							 , const NPVariant *args
							 , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 2)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer && handle the NULL case
  if (   ! NPVARIANT_IS_OBJECT(args[0])
	 || ! NPVARIANT_IS_STRING(args[1])
	 )
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: context_remove_application_action");
      return result;
    }

  // TODO retain object
  UnityWebappsContext *
    context = (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;
  
  char * label = create_safe_string_for (&args[1]);
  
  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  // TODO leak callback ... 
  unity_webapps_context_remove_application_action (context, label);
  
  g_free (label);

  return result;
}

NPVariant
unity_webapps_binding_context_set_application_accept_data (NPP instance
                                                           , NPObject * npobject
                                                           , const NPVariant *args
                                                           , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 2)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  if (   ! NPVARIANT_IS_OBJECT(args[0])
	 || ! NPVARIANT_IS_OBJECT(args[1])
	 )
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: context_set_application_accept_data");
      return result;
    }

  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  // TODO retain object
  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;

  NPVariant mimeTypesCountVariant;
  if (! NPN_GetProperty(instance
                        , NPVARIANT_TO_OBJECT(args[1])
                        , NPN_GetStringIdentifier("length")
                        , &mimeTypesCountVariant))
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: invalid argument (no length property)");
      return result;
    }
  if (! NPVARIANT_IS_DOUBLE(mimeTypesCountVariant))
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: invalid argument (invalid length property)");
      return result;
    }

  int mimeTypesCount = (int) NPVARIANT_TO_DOUBLE(mimeTypesCountVariant);
  NPN_ReleaseVariantValue(&mimeTypesCountVariant);
  if (mimeTypesCount == 0)
    {
      return result;
    }

  if (mimeTypesCount < 0)
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: invalid mime types count");
      return result;
    }

  // first pass to get the proper size to allocate
  size_t mimeTypesSize = 0;
  int mimeTypesIdx = 0;
  for ( ; mimeTypesIdx < mimeTypesCount
          ; ++mimeTypesIdx)
    {
      NPVariant curMimeType;
      if (! NPN_GetProperty(instance
                            , NPVARIANT_TO_OBJECT(args[1])
                            , NPN_GetIntIdentifier(mimeTypesIdx)
                            , &curMimeType))
        {
          break;
        }
      if (! NPVARIANT_IS_STRING(curMimeType))
        {
          break;
        }
      mimeTypesSize += NPVARIANT_TO_STRING(curMimeType).UTF8Length + 1;
      NPN_ReleaseVariantValue(&curMimeType);
  }
  if (mimeTypesIdx != mimeTypesCount)
    {
      NPN_SetException(npobject, "Malformed mime types object");
      return result;
    }

  char* pMimeTypes = g_malloc0(mimeTypesSize);

  size_t mimeTypesStructSize =
    sizeof(UnityWebappsStrWrapperDesc) * mimeTypesCount;
  UnityWebappsStrWrapperDesc* pMimeTypesStruct =
    g_malloc0(mimeTypesStructSize);

  char * pCurMimeTypes = pMimeTypes;
  for (mimeTypesIdx = 0
         ; mimeTypesIdx < mimeTypesCount
         ; ++mimeTypesIdx)
    {
      NPVariant curMimeType;
      if (! NPN_GetProperty(instance
                            , NPVARIANT_TO_OBJECT(args[1])
                            , NPN_GetIntIdentifier(mimeTypesIdx)
                            , &curMimeType))
        {
          break;
        }
      if (! NPVARIANT_IS_STRING(curMimeType))
        {
          break;
        }
      const size_t curMimeTypeSize = NPVARIANT_TO_STRING(curMimeType).UTF8Length + 1;

      pMimeTypesStruct[mimeTypesIdx].str = pCurMimeTypes;
      pCurMimeTypes += curMimeTypeSize;
      g_strlcpy((char *) pMimeTypesStruct[mimeTypesIdx].str
                , NPVARIANT_TO_STRING(curMimeType).UTF8Characters
                , curMimeTypeSize);
      NPN_ReleaseVariantValue(&curMimeType);
  }
  if (mimeTypesIdx != mimeTypesCount)
    {
      NPN_SetException(npobject, "Malformed mime types object");
      g_free(pMimeTypes);
      g_free(pMimeTypesStruct);
      return result;
    }

  unity_webapps_context_set_application_accept_data (context
                                                     , pMimeTypesStruct
                                                     , mimeTypesCount);

  g_free(pMimeTypes);
  g_free(pMimeTypesStruct);

  return result;
}

// TODO cleanup this mess
NPVariant
unity_webapps_binding_context_add_application_actions (NPP instance
                                                       , NPObject * npobject
                                                       , const NPVariant * args
                                                       , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 3)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer && handle the NULL case
  if (   ! NPVARIANT_IS_OBJECT(args[0])
	 || ! NPVARIANT_IS_OBJECT(args[1])
	 || ( ! NPVARIANT_IS_DOUBLE(args[2]) && ! NPVARIANT_IS_INT32(args[2]))
	 )
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: context_add_application_actions");
      return result;
    }

  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  // TODO retain object
  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;

  int actionsCount = 0;
  if (NPVARIANT_IS_INT32(args[2]))
    {
      actionsCount = NPVARIANT_TO_INT32(args[2]);
    }
  else if (NPVARIANT_IS_DOUBLE(args[2]))
    {
      actionsCount = NPVARIANT_TO_DOUBLE(args[2]);
    }

  if (actionsCount == 0)
    {
      return result;
    }
  
  if (actionsCount < 0)
    {
      NPN_SetException(npobject, "Invalid action count");
      return result;
    }
  
  // first pass to get the proper size to allocate
  size_t actionPathsSize = 0;
  int curActionIdx = 0;
  for ( ; curActionIdx < actionsCount
          ; ++curActionIdx)
    {
      NPVariant curAction;
      if (! NPN_GetProperty(instance
                            , NPVARIANT_TO_OBJECT(args[1])
                            , NPN_GetIntIdentifier(curActionIdx)
                            , &curAction))
        {
          break;
        }
      if (! NPVARIANT_IS_OBJECT(curAction))
        {
          break;
        }

      NPVariant actionPath;
      if (! NPN_GetProperty(instance
                            , NPVARIANT_TO_OBJECT(curAction)
                            , NPN_GetStringIdentifier("path")
                            , &actionPath))
        {
          NPN_ReleaseVariantValue(&curAction);
          break;
        }
      if (! NPVARIANT_IS_STRING(actionPath))
        {
          NPN_ReleaseVariantValue(&curAction);
          NPN_ReleaseVariantValue(&actionPath);
          break;
        }
      actionPathsSize += NPVARIANT_TO_STRING(actionPath).UTF8Length + 1;

      NPN_ReleaseVariantValue(&curAction);
      NPN_ReleaseVariantValue(&actionPath);
  }

  if (curActionIdx != actionsCount)
    {
      NPN_SetException(npobject, "Malformed action description object");
      return result;
    }

  size_t actionStructSize = sizeof(UnityWebappsApplicationActionDesc) * actionsCount;

  char* pActionPaths = g_malloc0(actionPathsSize);

  UnityWebappsApplicationActionDesc*
    pActions = g_malloc0(actionStructSize);

  char* pCurActionPath = pActionPaths;
  for(curActionIdx = 0
        ; curActionIdx < actionsCount
        ; ++curActionIdx)
    {
      NPVariant curAction;
      if (! NPN_GetProperty(instance
                            , NPVARIANT_TO_OBJECT(args[1])
                            , NPN_GetIntIdentifier(curActionIdx)
                            , &curAction))
        {
          break;
        }
      if (! NPVARIANT_IS_OBJECT(curAction))
        {
          break;
        }

      NPVariant actionPath;
      if (! NPN_GetProperty(instance
                            , NPVARIANT_TO_OBJECT(curAction)
                            , NPN_GetStringIdentifier("path")
                            , &actionPath))
        {
          NPN_ReleaseVariantValue(&curAction);
          break;
        }
      if (! NPVARIANT_IS_STRING(actionPath))
        {
          NPN_ReleaseVariantValue(&curAction);
          NPN_ReleaseVariantValue(&actionPath);
          break;
        }
      const size_t curActionPathSize = NPVARIANT_TO_STRING(actionPath).UTF8Length + 1;

      pActions[curActionIdx].path = pCurActionPath;
      pCurActionPath += curActionPathSize;
      g_strlcpy((char *) pActions[curActionIdx].path
                , NPVARIANT_TO_STRING(actionPath).UTF8Characters
                , curActionPathSize);
      NPN_ReleaseVariantValue(&actionPath);

      NPVariant actionCallback;
      if (! NPN_GetProperty(instance
                            , NPVARIANT_TO_OBJECT(curAction)
                            , NPN_GetStringIdentifier("callback")
                            , &actionCallback))
        {
          NPN_ReleaseVariantValue(&curAction);
          break;
        }
      if (! NPVARIANT_IS_OBJECT(actionCallback))
        {
          NPN_ReleaseVariantValue(&curAction);
          NPN_ReleaseVariantValue(&actionCallback);
          break;
        }
      NPObject * callback = NPVARIANT_TO_OBJECT(actionCallback);
      ADD_NPOBJECT_RETAIN_FOR_CONTEXT(instance, context, callback);

      // don't really care about the last param (simplification)

      NPObject * wrappedCallback =
        create_wrapped_callback_object_for (instance, callback);
      ADD_NPOBJECT_RETAIN_FOR_CONTEXT(instance, context, wrappedCallback);

      pActions[curActionIdx].callback = UnityWebappsContextActionCallback_dispatcher;
      pActions[curActionIdx].user_data = wrappedCallback;

      g_message ("adding action path name %s", pActions[curActionIdx].path);
      
      NPN_ReleaseVariantValue(&curAction);
    }
  if (curActionIdx != actionsCount)
    {
      // should not happen
      NPN_SetException(npobject, "Malformed action description object");
      
      g_free(pActionPaths);
      g_free(pActions);
      
      return result;
    }

  unity_webapps_context_add_application_actions (context
                                                 , pActions
                                                 , actionsCount);

  g_free(pActionPaths);
  g_free(pActions);

  return result;
}


NPVariant
unity_webapps_binding_context_remove_application_actions (NPP instance
							  , NPObject * npobject
							  , const NPVariant * args
							  , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 1)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer && handle the NULL case
  if (   ! NPVARIANT_IS_OBJECT(args[0]) )
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: context_remove_application_actions");
      return result;
    }
  
  // TODO retain object
  UnityWebappsContext *
    context = (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;
  
  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  unity_webapps_context_remove_application_actions (context);
  
  return result;
}




/**
 *	Music player
 *
 */

NPVariant
unity_webapps_binding_music_player_init (NPP instance
					 , NPObject * npobject
					 , const NPVariant *args
					 , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 2)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer
  if (   ! NPVARIANT_IS_OBJECT(args[0])
	 || ! NPVARIANT_IS_STRING(args[1]))
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: music_player_init");
      return result;
    }

  UnityWebappsContext * context = (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;
	
  gchar * title = create_safe_string_for (&args[1]);
	
  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  unity_webapps_music_player_init (context, title);

  g_free (title);

  return result;
}


void UnityWebappsMusicPlayerCallback_dispatcher (UnityWebappsContext *context, gpointer user_data)
{
  // not really safe ... 
  wrapped_callback_t * pCallbackObject = (wrapped_callback_t *) user_data;

  // fill out the arguments
  NPVariant args [2];

  // TODO call w/ meaningful values
  NULL_TO_NPVARIANT (args[0]); //context
  NULL_TO_NPVARIANT (args[1]); //user data

  NPVariant response;
	
  NPN_InvokeDefault (pCallbackObject->instance
                     , pCallbackObject->wrapped_callback
                     , args
                     , G_N_ELEMENTS(args)
                     , &response);

  NPN_ReleaseVariantValue(&response);
}


NPVariant
unity_webapps_binding_music_player_on_play_pause_callback (NPP instance
							   , NPObject * npobject
							   , const NPVariant *args
							   , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 3)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer && handle the NULL case
  if (   ! NPVARIANT_IS_OBJECT(args[0])
	 || ! NPVARIANT_IS_OBJECT(args[1])
	 || ( ! NPVARIANT_IS_OBJECT(args[2]) && ! is_null_or_void (args[2])) )
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: music_player_on_play_pause_callback");
      return result;
    }

  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;

  NPObject * callback = NPVARIANT_TO_OBJECT(args[1]);
  ADD_NPOBJECT_RETAIN_FOR_CONTEXT(instance, context, callback);
  
  // don't really care about the last param (simplification)
	
  NPObject * wrappedCallback =
    create_wrapped_callback_object_for (instance, callback);
  ADD_NPOBJECT_RETAIN_FOR_CONTEXT(instance, context, wrappedCallback);

  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  // TODO !!!!
  // mmmh make sure that it works fine (that the same dispatcher function can be registered multiple times)
  unity_webapps_music_player_on_play_pause_callback (context
                                                     , UnityWebappsMusicPlayerCallback_dispatcher
                                                     , wrappedCallback);

  return result;
}


NPVariant
unity_webapps_binding_music_player_on_previous_callback (NPP instance
							 , NPObject * npobject
							 , const NPVariant *args
							 , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 3)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer && handle the NULL case
  if (   ! NPVARIANT_IS_OBJECT(args[0])
	 || ! NPVARIANT_IS_OBJECT(args[1])
	 || ( ! NPVARIANT_IS_OBJECT(args[2]) && ! is_null_or_void (args[2])) )
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: music_player_on_previous_callback");
      return result;
    }

  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;
  
  NPObject * callback = NPVARIANT_TO_OBJECT(args[1]);
  ADD_NPOBJECT_RETAIN_FOR_CONTEXT(instance, context, callback);
  
  // don't really care about the last param (simplification)
  
  NPObject * wrappedCallback =
    create_wrapped_callback_object_for (instance, callback);
  ADD_NPOBJECT_RETAIN_FOR_CONTEXT(instance, context, wrappedCallback);

  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  // TODO !!!!
  // mmmh make sure that it works fine (that the same dispatcher function can be registered multiple times)
  unity_webapps_music_player_on_previous_callback (context
                                                   , UnityWebappsMusicPlayerCallback_dispatcher
                                                   , wrappedCallback);

  return result;
}


NPVariant
unity_webapps_binding_music_player_on_next_callback (NPP instance
						     , NPObject * npobject
						     , const NPVariant *args
						     , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 3)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer && handle the NULL case
  if (   ! NPVARIANT_IS_OBJECT(args[0])
	 || ! NPVARIANT_IS_OBJECT(args[1])
	 || ( ! NPVARIANT_IS_OBJECT(args[2]) && ! is_null_or_void (args[2])) )
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: music_player_on_next_callback");
      return result;
    }

  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;
	
  NPObject * callback = NPVARIANT_TO_OBJECT(args[1]);
  ADD_NPOBJECT_RETAIN_FOR_CONTEXT(instance, context, callback);
	
  // don't really care about the last param (simplification)
	
  NPObject * wrappedCallback =
    create_wrapped_callback_object_for (instance, callback);
  ADD_NPOBJECT_RETAIN_FOR_CONTEXT(instance, context, wrappedCallback);

  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  // TODO !!!!
  // mmmh make sure that it works fine (that the same dispatcher function can be registered multiple times)
  unity_webapps_music_player_on_next_callback (context
                                               , UnityWebappsMusicPlayerCallback_dispatcher
                                               , wrappedCallback);

  return result;
}


NPVariant
unity_webapps_binding_music_player_set_track (NPP instance
					      , NPObject * npobject
					      , const NPVariant *args
					      , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 5)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer
  if (   ! NPVARIANT_IS_OBJECT(args[0])
	 || ( ! NPVARIANT_IS_STRING(args[1]) && ! is_null_or_void (args[1]))
	 || ( ! NPVARIANT_IS_STRING(args[2]) && ! is_null_or_void (args[2]))
	 || ( ! NPVARIANT_IS_STRING(args[3]) && ! is_null_or_void (args[3]))
	 || ( ! NPVARIANT_IS_STRING(args[4]) && ! is_null_or_void (args[4]))
         )
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: music_player_set_track");
      return result;
    }
	
  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;
	
  gchar * artist = NULL;
  if (! is_null_or_void (args[1]))
    {
      artist = create_safe_string_for (&args[1]);
    }
	
  gchar * album = NULL;
  if (! is_null_or_void (args[2]))
    {
      album = create_safe_string_for (&args[2]);
    }
	
  gchar * title = NULL;
  if (! is_null_or_void (args[3]))
    {
      title = create_safe_string_for (&args[3]);
    }

  gchar * icon_url = NULL;
  if (! is_null_or_void (args[4]))
    {
      icon_url = create_safe_string_for (&args[4]);
    }
  
  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  unity_webapps_music_player_set_track (context, artist, album, title, icon_url);
	
  g_free (artist);
  g_free (album);
  g_free (title);
  g_free (icon_url);

  return result;
}


NPVariant
unity_webapps_binding_music_player_get_playback_state (NPP instance
						       , NPObject * npobject
						       , const NPVariant *args
						       , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 1)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer && handle the NULL case
  if ( ! NPVARIANT_IS_OBJECT(args[0]))
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: music_player_get_playback_state");
      return result;
    }

  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;

  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  INT32_TO_NPVARIANT ((int32_t) unity_webapps_music_player_get_playback_state (context), result);

  return result;
}

NPVariant
unity_webapps_binding_music_player_get_can_go_previous (NPP instance
							, NPObject * npobject
							, const NPVariant *args
							, uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);
	
  if (argCount != 1)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }
	
  // TODO must add proper type validation layer && handle the NULL case
  if ( ! NPVARIANT_IS_OBJECT(args[0]))
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: music_player_get_can_go_previous");
      return result;
    }
	
  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;
  
  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  BOOLEAN_TO_NPVARIANT ((bool) unity_webapps_music_player_get_can_go_previous (context), result);
	
  return result;
}

NPVariant
unity_webapps_binding_music_player_get_can_go_next (NPP instance
						    , NPObject * npobject
						    , const NPVariant *args
						    , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);
	
  if (argCount != 1)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }
	
  // TODO must add proper type validation layer && handle the NULL case
  if ( ! NPVARIANT_IS_OBJECT(args[0]))
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: music_player_get_can_go_next");
      return result;
    }
  
  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;
  
  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  BOOLEAN_TO_NPVARIANT ((bool) unity_webapps_music_player_get_can_go_next (context), result);
	
  return result;
}

NPVariant
unity_webapps_binding_music_player_get_can_play (NPP instance
						 , NPObject * npobject
						 , const NPVariant *args
						 , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);
	
  if (argCount != 1)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }
	
  // TODO must add proper type validation layer && handle the NULL case
  if ( ! NPVARIANT_IS_OBJECT(args[0]))
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: music_player_get_can_play");
      return result;
    }
	
  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;
  
  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  BOOLEAN_TO_NPVARIANT ((bool) unity_webapps_music_player_get_can_play (context), result);
	
  return result;
}

NPVariant
unity_webapps_binding_music_player_get_can_pause (NPP instance
						  , NPObject * npobject
						  , const NPVariant *args
						  , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);
	
  if (argCount != 1)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }
	
  // TODO must add proper type validation layer && handle the NULL case
  if ( ! NPVARIANT_IS_OBJECT(args[0]))
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: music_player_get_can_pause");
      return result;
    }
	
  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;
  
  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  BOOLEAN_TO_NPVARIANT ((bool) unity_webapps_music_player_get_can_pause (context), result);
  
  return result;
}


NPVariant
unity_webapps_binding_music_player_set_playback_state (NPP instance
						       , NPObject * npobject
						       , const NPVariant *args
						       , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);
	
  if (argCount != 2)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }
	
  // TODO must add proper type validation layer && handle the NULL case
  if ( ! NPVARIANT_IS_OBJECT(args[0])
       || ( ! NPVARIANT_IS_BOOLEAN(args[1]) && ! NPVARIANT_IS_INT32(args[1]) && ! NPVARIANT_IS_DOUBLE(args[1]) ) )
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: music_player_set_playback_state");
      return result;
    }
  
  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;
  
  // TODO should check the value against a max state or invalid state (or delegate to function)
  UnityWebappsMusicPlayerPlaybackState state =
    UNITY_WEBAPPS_MUSIC_PLAYER_PLAYBACK_STATE_PLAYING;
  if (NPVARIANT_IS_INT32(args[1]))
    {
      state = (NPVARIANT_TO_INT32(args[1]) != 0)
        ? UNITY_WEBAPPS_MUSIC_PLAYER_PLAYBACK_STATE_PAUSED
        : UNITY_WEBAPPS_MUSIC_PLAYER_PLAYBACK_STATE_PLAYING;
    }
  else if (NPVARIANT_IS_BOOLEAN(args[1]))
    {
      state = (NPVARIANT_TO_BOOLEAN(args[1]) != 0)
        ? UNITY_WEBAPPS_MUSIC_PLAYER_PLAYBACK_STATE_PAUSED
        : UNITY_WEBAPPS_MUSIC_PLAYER_PLAYBACK_STATE_PLAYING;
    }
  else
    {
      state = (NPVARIANT_TO_DOUBLE(args[1]) != 0.0)
        ? UNITY_WEBAPPS_MUSIC_PLAYER_PLAYBACK_STATE_PAUSED
        : UNITY_WEBAPPS_MUSIC_PLAYER_PLAYBACK_STATE_PLAYING;
    }

  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  unity_webapps_music_player_set_playback_state (context, state);
  
  return result;
}

NPVariant
unity_webapps_binding_music_player_set_can_go_previous (NPP instance
							, NPObject * npobject
							, const NPVariant *args
							, uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);
	
  if (argCount != 2)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }
	
  // TODO must add proper type validation layer && handle the NULL case
  if ( ! NPVARIANT_IS_OBJECT(args[0])
       || ! is_boolean_convertible_value(args[1])
       )
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: music_player_set_can_go_previous");
      return result;
    }
	
  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;
  
  // TODO should check the value against a max state or invalid state (or delegate to function)
  gboolean can_go_previous = extract_boolean_value_from(args[1]);

  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  unity_webapps_music_player_set_can_go_previous (context, can_go_previous);

  return result;
}

NPVariant
unity_webapps_binding_music_player_set_can_go_next (NPP instance
						    , NPObject * npobject
						    , const NPVariant *args
						    , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);
	
  if (argCount != 2)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }
	
  // TODO must add proper type validation layer && handle the NULL case
  if ( ! NPVARIANT_IS_OBJECT(args[0])
       || ! is_boolean_convertible_value(args[1])
       )
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: music_player_set_can_go_next");
      return result;
    }
	
  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;
  
  // TODO should check the value against a max state or invalid state (or delegate to function)
  gboolean can_go_next = extract_boolean_value_from(args[1]);

  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  unity_webapps_music_player_set_can_go_next (context, can_go_next);
	
  return result;
}

NPVariant
unity_webapps_binding_music_player_set_can_play (NPP instance
						 , NPObject * npobject
						 , const NPVariant *args
						 , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);
	
  if (argCount != 2)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }
	
  // TODO must add proper type validation layer && handle the NULL case
  if ( ! NPVARIANT_IS_OBJECT(args[0])
       && ! is_boolean_convertible_value(args[1])
       )
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: music_player_set_can_play");
      return result;
    }
	
  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;

  // TODO should check the value against a max state or invalid state (or delegate to function)
  gboolean can_go_play = extract_boolean_value_from(args[1]);

  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  unity_webapps_music_player_set_can_play (context, can_go_play);
	
  return result;
}

NPVariant
unity_webapps_binding_music_player_set_can_pause (NPP instance
						  , NPObject * npobject
						  , const NPVariant *args
						  , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);
	
  if (argCount != 2)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }
	
  // TODO must add proper type validation layer && handle the NULL case
  if ( ! NPVARIANT_IS_OBJECT(args[0])
       || ! is_boolean_convertible_value(args[1])
       )
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: music_player_set_can_pause");
      return result;
    }
  
  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;
  
  // TODO should check the value against a max state or invalid state (or delegate to function)
  gboolean can_go_pause = extract_boolean_value_from(args[1]);

  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  unity_webapps_music_player_set_can_pause (context, can_go_pause);
	
  return result;
}




/**
 *  Permissions
 *
 */

NPVariant
    unity_webapps_binding_permissions_is_integration_allowed (NPP instance
         , NPObject * npobject
         , const NPVariant *args
         , uint32_t argCount)
{
    NPVariant result;
    NULL_TO_NPVARIANT (result);

  if (argCount != 0)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call: permissions_is_integration_allowed");
      return result;
    }

    REACHED_UNITY_WEBAPPS_FUNC_CALL();
    gboolean isallowed = unity_webapps_permissions_is_integration_allowed();

    BOOLEAN_TO_NPVARIANT(isallowed, result);

    return result;
}

NPVariant
unity_webapps_binding_permissions_get_domain_allowed (NPP instance
						      , NPObject * npobject
						      , const NPVariant *args
						      , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 1)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer
  if ( ! NPVARIANT_IS_STRING(args[0]))
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: permissions_get_domain_allowed");
      return result;
    }
  
  gchar * domain = create_safe_string_for (&args[0]);
  
  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  gboolean isallowed = unity_webapps_permissions_get_domain_allowed (domain);

  BOOLEAN_TO_NPVARIANT(isallowed, result);

  g_free (domain);

  return result;
}


NPVariant
unity_webapps_binding_permissions_get_domain_preauthorized (NPP instance
                                                            , NPObject * npobject
                                                            , const NPVariant *args
                                                            , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 1)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer
  if ( ! NPVARIANT_IS_STRING(args[0]))
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: permissions_get_domain_preauthorized");
      return result;
    }
  
  gchar * domain = create_safe_string_for (&args[0]);
  
  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  gboolean ispreauthorized = unity_webapps_permissions_get_domain_preauthorized (domain);

  BOOLEAN_TO_NPVARIANT(ispreauthorized, result);

  g_free (domain);

  return result;
}

NPVariant
unity_webapps_binding_permissions_allow_domain (NPP instance
						, NPObject * npobject
						, const NPVariant *args
						, uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 1)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer
  if ( ! NPVARIANT_IS_STRING(args[0]))
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: permissions_allow_domain");
      return result;
    }
	
  gchar * domain = create_safe_string_for (&args[0]);
	
  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  unity_webapps_permissions_allow_domain (domain);

  g_free (domain);

  return result;
}

NPVariant
unity_webapps_binding_permissions_get_domain_dontask (NPP instance
						      , NPObject * npobject
						      , const NPVariant *args
						      , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 1)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer
  if ( ! NPVARIANT_IS_STRING(args[0]))
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: permissions_get_domain_dontask");
      return result;
    }
	
  gchar * domain = create_safe_string_for (&args[0]);
  
  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  gboolean dontask = unity_webapps_permissions_get_domain_dontask (domain);
  
  BOOLEAN_TO_NPVARIANT (dontask, result);
  
  g_free (domain);

  return result;
}

NPVariant
unity_webapps_binding_permissions_dontask_domain (NPP instance
						  , NPObject * npobject
						  , const NPVariant *args
						  , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 1)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer
  if ( ! NPVARIANT_IS_STRING(args[0]))
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: permissions_dontask_domain");
      return result;
    }

  gchar * domain = create_safe_string_for (&args[0]);
	
  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  unity_webapps_permissions_dontask_domain (domain);

  g_free (domain);

  return result;
}




/**
 *	Notification
 *
 */

NPVariant
unity_webapps_binding_notification_show_notification (NPP instance
						      , NPObject * npobject
						      , const NPVariant *args
						      , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  // TODO fix that, automatically infer the mandatory argument count, etc.
  if (argCount != 4)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer
  if (   ! NPVARIANT_IS_OBJECT(args[0])
	 || ! NPVARIANT_IS_STRING(args[1])
	 || ( ! NPVARIANT_IS_STRING(args[2]) && ! is_null_or_void (args[2]))
	 || ( ! NPVARIANT_IS_STRING(args[3]) && ! is_null_or_void (args[3])) )
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: notification_show_notification");
      return result;
    }

  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;

  gchar * summary = create_safe_string_for (&args[1]);
  gchar * body = NULL;
  if (NPVARIANT_IS_STRING(args[2]))
    {
      body = create_safe_string_for (&args[2]);
    }
  gchar * icon_url = NULL;
  if (NPVARIANT_IS_STRING(args[3]))
    {
      icon_url = create_safe_string_for (&args[3]);
    }

  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  unity_webapps_notification_show_notification (context, summary, body, icon_url);

  g_free (summary);
  g_free (body);
  g_free (icon_url);

  return result;
}



/**
 * Launcher
 *
 *
 */

NPVariant
unity_webapps_binding_launcher_set_count (NPP instance
					  , NPObject * npobject
					  , const NPVariant *args
					  , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 2)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer
  if (   ! NPVARIANT_IS_OBJECT(args[0])
	 || (! NPVARIANT_IS_INT32(args[1]) && ! NPVARIANT_IS_DOUBLE(args[1]))
         )
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: launcher_set_count");
      return result;
    }
	
  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;
	
  gint count = 0;
  if (NPVARIANT_IS_INT32(args[1]))
    {
      count = NPVARIANT_TO_INT32(args[1]);
    }
  else if (NPVARIANT_IS_DOUBLE(args[1]))
    {
      count = NPVARIANT_TO_DOUBLE(args[1]);
    }

  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  unity_webapps_launcher_set_count (context, count);

  return result;
}


NPVariant
unity_webapps_binding_launcher_clear_count (NPP instance
					    , NPObject * npobject
					    , const NPVariant *args
					    , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 1)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer
  if ( ! NPVARIANT_IS_OBJECT(args[0]))
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: launcher_clear_count");
      return result;
    }
	
  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;
	
  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  unity_webapps_launcher_clear_count (context);

  return result;
}

NPVariant
unity_webapps_binding_launcher_set_progress (NPP instance
					     , NPObject * npobject
					     , const NPVariant *args
					     , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 2)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer
  if (   ! NPVARIANT_IS_OBJECT(args[0])
	 || ! NPVARIANT_IS_DOUBLE(args[1]))
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: launcher_set_progress");
      return result;
    }
	
  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;
  
  gdouble progress = NPVARIANT_TO_DOUBLE(args[1]);
  
  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  unity_webapps_launcher_set_progress (context, progress);

  return result;
}

NPVariant
unity_webapps_binding_launcher_clear_progress (NPP instance
					       , NPObject * npobject
					       , const NPVariant *args
					       , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 1)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer
  if ( ! NPVARIANT_IS_OBJECT(args[0]))
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: launcher_clear_progress");
      return result;
    }
	
  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;
	
  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  unity_webapps_launcher_clear_progress (context);

  return result;
}

NPVariant
unity_webapps_binding_launcher_set_urgent (NPP instance
					   , NPObject * npobject
					   , const NPVariant *args
					   , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 1)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer
  if ( ! NPVARIANT_IS_OBJECT(args[0]))
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: launcher_set_urgent");
      return result;
    }
	
  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;
	
  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  unity_webapps_launcher_set_urgent (context);

  return result;
}

void
UnityWebappsLauncherCallback_dispatcher (UnityWebappsContext *context
					 , gpointer user_data)
{
  // not really safe ... 
  wrapped_callback_t * pCallbackObject = (wrapped_callback_t *) user_data;
	
  // fill out the arguments
  NPVariant args [2];
	
  // TODO call w/ meaningful values
  NULL_TO_NPVARIANT (args[0]);
  NULL_TO_NPVARIANT (args[1]);

  NPVariant response;

  NPN_InvokeDefault (pCallbackObject->instance
		     , pCallbackObject->wrapped_callback
		     , args
		     , G_N_ELEMENTS(args)
		     , &response);

  NPN_ReleaseVariantValue (&response);
}


NPVariant
unity_webapps_binding_launcher_add_action (NPP instance
					   , NPObject * npobject
					   , const NPVariant *args
					   , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 4)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }
	
  // TODO must add proper type validation layer && handle the NULL case
  if (   ! NPVARIANT_IS_OBJECT(args[0])
	 || ! NPVARIANT_IS_STRING(args[1])
	 || ! NPVARIANT_IS_OBJECT(args[2])
	 || ( ! NPVARIANT_IS_OBJECT(args[3]) && ! is_null_or_void (args[3])) )
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;

  gchar * label = create_safe_string_for (&args[1]);

  NPObject * callback = NPVARIANT_TO_OBJECT(args[2]);
  ADD_NPOBJECT_RETAIN_FOR_CONTEXT(instance, context, callback);
	
  // don't really care about the last param (simplification)
	
  NPObject * wrappedCallback =
    create_wrapped_callback_object_for (instance, callback);
  ADD_NPOBJECT_RETAIN_FOR_CONTEXT(instance, context, wrappedCallback);

  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  // TODO !!!!
  // mmmh make sure that it works fine (that the same dispatcher function can be registered multiple times)
  unity_webapps_launcher_add_action (context
				     , label
				     , UnityWebappsLauncherCallback_dispatcher
				     , wrappedCallback);

  g_free (label);

  return result;
}

NPVariant
unity_webapps_binding_launcher_remove_action (NPP instance
					      , NPObject * npobject
					      , const NPVariant *args
					      , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 2)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }
  
  // TODO must add proper type validation layer
  if (   ! NPVARIANT_IS_OBJECT(args[0])
	 || ! NPVARIANT_IS_STRING(args[1]))
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: launcher_remove_action");
      return result;
    }
  
  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;
  gchar * label = create_safe_string_for (&args[1]);
  
  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  unity_webapps_launcher_remove_action (context, label);
  
  g_free (label);
  
  return result;
}

NPVariant
unity_webapps_binding_launcher_remove_actions (NPP instance
					       , NPObject * npobject
					       , const NPVariant *args
					       , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 1)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer
  if ( ! NPVARIANT_IS_OBJECT(args[0]))
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: launcher_remove_actions");
      return result;
    }
	
  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;
	
  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  unity_webapps_launcher_remove_actions (context);

  return result;
}




/**
 * Message indicator
 *
 *
 */

void UnityWebappsIndicatorCallback_dispatcher (UnityWebappsContext *context, gpointer user_data)
{
  // not really safe ... 
  wrapped_callback_t * pCallbackObject = (wrapped_callback_t *) user_data;
	
  // fill out the arguments
  NPVariant args [2];
	
  // TODO call w/ meaningful values
  NULL_TO_NPVARIANT (args[0]);
  NULL_TO_NPVARIANT (args[1]);
	
  NPVariant response;
	
  NPN_InvokeDefault (pCallbackObject->instance
                     , pCallbackObject->wrapped_callback
                     , args
                     , G_N_ELEMENTS(args)
                     , &response);

  NPN_ReleaseVariantValue (&response);
}

NPVariant
unity_webapps_binding_indicator_show_indicator (NPP instance
						, NPObject * npobject
						, const NPVariant *args
                            			, uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 2)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer
  if (   ! NPVARIANT_IS_OBJECT(args[0])
	 || ! NPVARIANT_IS_STRING(args[1]))
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }
	
  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;

  gchar * name = create_safe_string_for (&args[1]);

  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  unity_webapps_indicator_show_indicator (context, name);

  g_free (name);

  return result;
}

NPVariant
unity_webapps_binding_indicator_clear_indicator (NPP instance
						 , NPObject * npobject
						 , const NPVariant *args
						 , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 2)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer
  if (   ! NPVARIANT_IS_OBJECT(args[0])
	 || ! NPVARIANT_IS_STRING(args[1]))
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: indicator_clear_indicator");
      return result;
    }

  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;

  gchar * name = create_safe_string_for (&args[1]);

  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  unity_webapps_indicator_clear_indicator (context, name);

  g_free ((gpointer) name);

  return result;
}


NPVariant
unity_webapps_binding_indicator_clear_indicators (NPP instance
						  , NPObject * npobject
						  , const NPVariant *args
						  , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);
  
  if (argCount != 1)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer
  if ( ! NPVARIANT_IS_OBJECT(args[0]) )
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: indicator_clear_indicators");
      return result;
    }
	
  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;
  
  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  unity_webapps_indicator_clear_indicators (context);
  
  return result;
}


NPVariant
unity_webapps_binding_indicator_set_callback (NPP instance
					      , NPObject * npobject
					      , const NPVariant *args
					      , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 4)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer && handle the NULL case
  if (   ! NPVARIANT_IS_OBJECT(args[0])
	 || ! NPVARIANT_IS_STRING(args[1])
	 || ! NPVARIANT_IS_OBJECT(args[2])
	 || ( ! NPVARIANT_IS_OBJECT(args[3]) && ! is_null_or_void (args[3])) )
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: indicator_set_callback");
      return result;
    }

  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;

  gchar * name = create_safe_string_for (&args[1]);

  NPObject * callback = NPVARIANT_TO_OBJECT(args[2]);
  ADD_NPOBJECT_RETAIN_FOR_CONTEXT(instance, context, callback);

  // don't really care about the last param (simplification)
	
  NPObject * wrappedCallback =
    create_wrapped_callback_object_for (instance, callback);
  ADD_NPOBJECT_RETAIN_FOR_CONTEXT(instance, context, wrappedCallback);

  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  // TODO !!!!
  // mmmh make sure that it works fine (that the same dispatcher function can be registered multiple times)
  unity_webapps_indicator_set_callback (context
					, name
					, UnityWebappsIndicatorCallback_dispatcher
					, wrappedCallback);

  g_free (name);

  return result;
}

NPVariant
unity_webapps_binding_indicator_set_property (NPP instance
					      , NPObject * npobject
					      , const NPVariant *args
					      , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 4)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer
  if (   ! NPVARIANT_IS_OBJECT(args[0])
	 || ! NPVARIANT_IS_STRING(args[1])
	 || ! NPVARIANT_IS_STRING(args[2])
	 || ! NPVARIANT_IS_STRING(args[3])
	 )
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: indicator_set_property");
      return result;
    }
	
  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;
	
  gchar * name = create_safe_string_for (&args[1]);
  gchar * property = create_safe_string_for (&args[2]);
  gchar * value = create_safe_string_for (&args[3]);

  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  unity_webapps_indicator_set_property (context, name, property, value);

  g_free (name);
  g_free (property);
  g_free (value);
	
  return result;
}

NPVariant
unity_webapps_binding_indicator_set_property_icon (NPP instance
						   , NPObject * npobject
						   , const NPVariant *args
						   , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 4)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer
  if (   ! NPVARIANT_IS_OBJECT(args[0])
	 || ! NPVARIANT_IS_STRING(args[1])
	 || ! NPVARIANT_IS_STRING(args[2])
	 || ! NPVARIANT_IS_STRING(args[3])
	 )
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: indicator_set_property_icon");
      return result;
    }

  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;
	
  gchar * name = create_safe_string_for (&args[1]);
  gchar * property = create_safe_string_for (&args[2]);
  gchar * icon_url = create_safe_string_for (&args[3]);

  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  unity_webapps_indicator_set_property_icon (context, name, property, icon_url);

  g_free (name);
  g_free (property);
  g_free (icon_url);

  return result;
}

NPVariant
unity_webapps_binding_indicator_add_action (NPP instance
					    , NPObject * npobject
					    , const NPVariant *args
					    , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 4)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer && handle the NULL case
  if (   ! NPVARIANT_IS_OBJECT(args[0])
	 || ! NPVARIANT_IS_STRING(args[1])
	 || ! NPVARIANT_IS_OBJECT(args[2])
	 || ( ! NPVARIANT_IS_OBJECT(args[3]) && ! is_null_or_void (args[3])) )
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: indicator_add_action");
      return result;
    }

  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;
	
  gchar * label = create_safe_string_for (&args[1]);
  
  NPObject * callback = NPVARIANT_TO_OBJECT(args[2]);
  ADD_NPOBJECT_RETAIN_FOR_CONTEXT(instance, context, callback);

  // don't really care about the last param (simplification)

  NPObject * wrappedCallback =
    create_wrapped_callback_object_for (instance, callback);
  ADD_NPOBJECT_RETAIN_FOR_CONTEXT(instance, context, wrappedCallback);

  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  // TODO !!!!
  // mmmh make sure that it works fine (that the same dispatcher function can be registered multiple times)
  unity_webapps_indicator_add_action (context,
				      label,
				      UnityWebappsIndicatorCallback_dispatcher,
				      wrappedCallback);

  g_free (label);

  return result;
}

NPVariant
unity_webapps_binding_indicator_get_presence (NPP instance
					      , NPObject * npobject
					      , const NPVariant *args
					      , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 1)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer
  if ( ! NPVARIANT_IS_OBJECT(args[0]))
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: indicator_get_presence");
      return result;
    }
	
  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;
  
  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  gchar * presence = unity_webapps_indicator_get_presence (context);
  if (NULL != presence)
    {
      // TODO strlen -> gnome utf8
      gint size = strlen(presence) + 1;
      gchar * ret = NPN_MemAlloc (size);
      if (NULL == ret)
        {
          NPN_SetException(npobject, "Could not allocate proper memory for call to: context_get_name");
          return result;
        }
      memset (ret, 0, size);
      strncpy (ret, presence, size);
      STRINGZ_TO_NPVARIANT(ret, result);
    }
  
  return result;
}

NPVariant
unity_webapps_binding_indicator_on_presence_changed_callback (NPP instance
							      , NPObject * npobject
							      , const NPVariant *args
							      , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 3)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer && handle the NULL case
  if (   ! NPVARIANT_IS_OBJECT(args[0])
	 || ! NPVARIANT_IS_OBJECT(args[1])
	 || ( ! NPVARIANT_IS_OBJECT(args[2]) && ! is_null_or_void (args[2])) )
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: indicator_on_presence_changed_callback");
      return result;
    }

  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;
	
  NPObject * callback = NPVARIANT_TO_OBJECT(args[1]);
  ADD_NPOBJECT_RETAIN_FOR_CONTEXT(instance, context, callback);

  NPObject * wrappedCallback =
    create_wrapped_callback_object_for (instance, callback);
  ADD_NPOBJECT_RETAIN_FOR_CONTEXT(instance, context, wrappedCallback);

  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  // TODO !!!!
  // mmmh make sure that it works fine (that the same dispatcher function can be registered multiple times)
  unity_webapps_indicator_on_presence_changed_callback (context,
							UnityWebappsIndicatorCallback_dispatcher,
							wrappedCallback);

  return result;
}

NPVariant
unity_webapps_binding_context_set_homepage (NPP instance
					    , NPObject * npobject
					    , const NPVariant *args
					    , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 2)
    {
      NPN_SetException(npobject, "Invalid argument count in NPAPI function call");
      return result;
    }

  // TODO must add proper type validation layer && handle the NULL case
  if (   ! NPVARIANT_IS_OBJECT(args[0])
	 || ! NPVARIANT_IS_STRING(args[1]))
    {
      NPN_SetException(npobject, "Invalid argument type in NPAPI function call: context_set_homepage");
      return result;
    }

  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;
	
  gchar * homepage = create_safe_string_for (&args[1]);
	
  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  unity_webapps_context_set_homepage (context, homepage);

  g_free ((gpointer) homepage);

  return result;
}

NPVariant
unity_webapps_binding_service_set_xid_for_browser_window_id (NPP instance
							     , NPObject * object
							     , const NPVariant *args
							     , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);
  
  if (argCount != 3)
    {
      NPN_SetException(object, "Invalid argument count in NPAPI function call");
      return result;
    }
  
  // TODO must add proper type validation layer && handle the NULL case
  if (   ! NPVARIANT_IS_OBJECT(args[0])
	 || ! NPVARIANT_IS_OBJECT(args[1])
	 || (! NPVARIANT_IS_INT32(args[2]) && ! NPVARIANT_IS_DOUBLE(args[2]))
	 )
    {
      NPN_SetException(object, "Invalid argument type in NPAPI function call: service_set_xid");
      return result;
    }
  
  UnityWebappsService * service =
    (UnityWebappsService *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;
  
  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[1].value.objectValue)->pWrapped;

  int window_id = 0;
  if (NPVARIANT_IS_INT32(args[2]))
    {
      window_id = NPVARIANT_TO_INT32(args[2]);
    }
  else if (NPVARIANT_IS_DOUBLE(args[2]))
    {
      window_id = NPVARIANT_TO_DOUBLE(args[2]);
    }
  
  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  unity_npapi_tools_set_xid_for_browser_window_id (context, window_id);

  return result;
}


NPVariant
unity_webapps_binding_context_get_interest_id (NPP instance
                                               , NPObject * object
                                               , const NPVariant *args
                                               , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 1)
    {
      NPN_SetException(object, "Invalid argument count in NPAPI function call");
      return result;
    }

  if ( ! NPVARIANT_IS_OBJECT(args[0]))
    {
      NPN_SetException(object, "Invalid argument type in NPAPI function call: context_get_interest_id");
      return result;
    }

  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;

  gint interest_id = unity_webapps_context_get_interest_id (context);

  INT32_TO_NPVARIANT(interest_id, result);

  return result;
}

NPVariant
unity_webapps_binding_context_get_name (NPP instance
                                        , NPObject * object
                                        , const NPVariant *args
                                        , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 1)
    {
      NPN_SetException(object, "Invalid argument count in NPAPI function call");
      return result;
    }

  if ( ! NPVARIANT_IS_OBJECT(args[0]))
    {
      NPN_SetException(object, "Invalid argument type in NPAPI function call: context_get_name");
      return result;
    }

  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;

  const gchar * name = unity_webapps_context_get_name (context);
  if (NULL != name)
    {
      gint size = strlen(name) + 1;
      gchar * ret = NPN_MemAlloc (size);
      if (NULL == ret)
        {
          NPN_SetException(object, "Could not allocate proper memory for call to: context_get_name");
          return result;
        }
      memset (ret, 0, size);
      strncpy (ret, name, size);
      STRINGZ_TO_NPVARIANT(ret, result);
    }

  return result;
}

NPVariant
unity_webapps_binding_context_get_domain (NPP instance
                                          , NPObject * object
                                          , const NPVariant *args
                                          , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);

  if (argCount != 1)
    {
      NPN_SetException(object, "Invalid argument count in NPAPI function call");
      return result;
    }

  if ( ! NPVARIANT_IS_OBJECT(args[0]))
    {
      NPN_SetException(object, "Invalid argument type in NPAPI function call: context_get_domain");
      return result;
    }

  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  UnityWebappsContext * context =
    (UnityWebappsContext *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;

  const gchar * domain = unity_webapps_context_get_domain (context);
  if (NULL != domain)
    {
      gint size = strlen(domain) + 1;
      gchar * ret = NPN_MemAlloc (size);
      if (NULL == ret)
        {
          NPN_SetException(object, "Could not allocate proper memory for call to: context_get_name");
          return result;
        }
      memset (ret, 0, size);
      strncpy (ret, domain, size);
      STRINGZ_TO_NPVARIANT(ret, result);
    }

  return result;
}

NPVariant
unity_webapps_binding_service_destroy_interest_for_context (NPP instance
                                                            , NPObject * object
                                                            , const NPVariant *args
                                                            , uint32_t argCount)
{
  NPVariant result;
  NULL_TO_NPVARIANT (result);
 
  if (argCount != 5)
    {
      NPN_SetException(object, "Invalid argument count in NPAPI function call");
      return result;
    }

  if ( ! NPVARIANT_IS_OBJECT(args[0])
       || ! NPVARIANT_IS_STRING(args[1])
       || ! NPVARIANT_IS_STRING(args[2])
       || (! NPVARIANT_IS_INT32(args[3]) && ! NPVARIANT_IS_DOUBLE(args[3]))
       || ! is_boolean_convertible_value(args[4])
       )
    {
      NPN_SetException(object, "Invalid argument type in NPAPI function call: service_destroy_interest_for_context");
      return result;
    }

  REACHED_UNITY_WEBAPPS_FUNC_CALL();

  UnityWebappsService * service =
    (UnityWebappsService *) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;

  const gchar *name = create_safe_string_for(&args[1]);

  const gchar *domain = create_safe_string_for(&args[2]);

  gint interest_id = -1;
  if (NPVARIANT_IS_INT32(args[3]))
    {
      interest_id = NPVARIANT_TO_INT32(args[3]);
    }
  else if (NPVARIANT_IS_DOUBLE(args[3]))
    {
      interest_id = (gint) NPVARIANT_TO_DOUBLE(args[3]);
    }

  gboolean abandoned = extract_boolean_value_from(args[4]);

  unity_webapps_service_destroy_interest_for_context (service
                                                      , name
                                                      , domain
                                                      , interest_id
                                                      , abandoned);

  g_free((gpointer) name);
  g_free((gpointer) domain);

  return result;
}

NPVariant
    unity_webapps_binding_launcher_add_static_action (NPP instance
         , NPObject * npobject
         , const NPVariant *args
         , uint32_t argCount)
{
    NPVariant result;
    NULL_TO_NPVARIANT (result);
    
    if (argCount != 3)
    {
        NPN_SetException (npobject, "Invalid number of arguments for function call");
	return result;
    }
    
    
    if (  ( ! NPVARIANT_IS_OBJECT(args[0]) && ! is_null_or_void(args[0]) ) || 
	 ( ! NPVARIANT_IS_STRING(args[1]) && ! is_null_or_void(args[1]) ) || 
	 ( ! NPVARIANT_IS_STRING(args[2]) && ! is_null_or_void(args[2]) ) )
    {
        NPN_SetException(npobject, "Invalid argument type for function call");
        return result;
    }
    
    REACHED_UNITY_WEBAPPS_FUNC_CALL();
    	UnityWebappsContext * context = NULL;

    if ( ! is_null_or_void (args[0]))
    {
    	context = (void*) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;
    }
    
	const gchar * label = "";

    if ( ! is_null_or_void (args[1]))
    {
    	label = create_safe_string_for (&args[1]);
    }
    
	const gchar * page = "";

    if ( ! is_null_or_void (args[2]))
    {
    	page = create_safe_string_for (&args[2]);
    }
    
unity_webapps_launcher_add_static_action(context, label, page);

    	g_free ((gpointer) label);
	g_free ((gpointer) page);
    return result;
}
    

NPVariant
    unity_webapps_binding_launcher_remove_static_actions (NPP instance
         , NPObject * npobject
         , const NPVariant *args
         , uint32_t argCount)
{
    NPVariant result;
    NULL_TO_NPVARIANT (result);
    
    if (argCount != 1)
    {
        NPN_SetException (npobject, "Invalid number of arguments for function call");
	return result;
    }
    
    
    if (  ( ! NPVARIANT_IS_OBJECT(args[0]) && ! is_null_or_void(args[0]) ) )
    {
        NPN_SetException(npobject, "Invalid argument type for function call");
        return result;
    }
    
    	UnityWebappsContext * context = NULL;

    REACHED_UNITY_WEBAPPS_FUNC_CALL();

    if ( ! is_null_or_void (args[0]))
    {
    	context = (void*) ((wrapped_void_ptr_t *) args[0].value.objectValue)->pWrapped;
    }
    
unity_webapps_launcher_remove_static_actions(context);
    
    return result;
}
