/*
 * Copyright (C) 2023, KylinSoft Co., Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 */
#ifndef POWER_H
#define POWER_H

#include <QObject>
#include <QtPlugin>
#include <QStyledItemDelegate>
#include <QGSettings>
#include <QDBusInterface>
#include <QDBusReply>
#include <QRadioButton>
#include <QButtonGroup>
#include <QApplication>

#include <ukcc/interface/interface.h>
#include <ukcc/widgets/titlelabel.h>
#include <ukcc/widgets/comboxframe.h>
#include <ukcc/widgets/fixlabel.h>

#ifdef KY_SDK_SYSINFO
#include <kysdk/kysdk-system/libkysysinfo.h>
#endif

#include <kysdk/kysdk-base/libkydiagnostics.h>
#include "kswitchbutton.h"

using namespace kdk;

namespace Ui {
class Power;
}

enum policy
{
    power_policy_performance = 0,
    power_policy_balance,
    power_policy_powersave,
};

class Power : public QObject, CommonInterface
{
    Q_OBJECT
    Q_PLUGIN_METADATA(IID "org.ukcc.CommonInterface")
    Q_INTERFACES(CommonInterface)

public:
    explicit Power();
    ~Power();

    QString plugini18nName()        Q_DECL_OVERRIDE;
    int pluginTypes()               Q_DECL_OVERRIDE;
    QWidget *pluginUi()             Q_DECL_OVERRIDE;
    bool isEnable() const           Q_DECL_OVERRIDE;
    const QString name() const      Q_DECL_OVERRIDE;
    bool isShowOnHomePage() const   Q_DECL_OVERRIDE;
    QIcon icon() const              Q_DECL_OVERRIDE;
    QString translationPath() const Q_DECL_OVERRIDE;

private:
    QString m_pluginName;
    int m_pluginType;
    bool m_firstLoad;

    QGSettings *m_powerManagerSettings = nullptr;
    QGSettings *m_styleSettings = nullptr;
    QGSettings *m_screenSettings = nullptr;
    QGSettings *m_procManagerSettings = nullptr;

    bool m_isWayland = false;
    bool m_isTablet = false;
    bool m_canSuspend = false;
    bool m_canHibernate = false;
    bool m_hasBattery = false;
    bool m_hasLid = false;
    bool m_batteryOnline = false;

    QWidget *m_pluginWidget;

#define defineMainWidgetFrame(frameName)   \
    TitleLabel *m_##frameName##TitleLabel; \
    QFrame *m_##frameName##Frame;          \
    QVBoxLayout *m_##frameName##Layout;    \
    void frameName##SettingsFrame(QWidget *)

#define defineButtonWidgetFrame(frameName) \
    QFrame *m_##frameName##Frame;          \
    QLabel *m_##frameName##Label;          \
    KSwitchButton *m_##frameName##Btn;     \
    QFrame *m_##frameName##Line

#define defineComboBoxWidgetFrame(frameName)    \
    QFrame *m_##frameName##Frame;               \
    QLabel *m_##frameName##Label;               \
    QComboBox *m_##frameName##ComboBox;         \
    QFrame *m_##frameName##Line;                \
    QStringList m_##frameName##Options

    defineMainWidgetFrame(generalPower);
    defineButtonWidgetFrame(wakeUpComputerNeedPwd);
    defineButtonWidgetFrame(wakeUpDisplayNeedPwd);
    defineComboBoxWidgetFrame(powerKeyAction);
    defineComboBoxWidgetFrame(idleTurnOffDisplay);
    defineComboBoxWidgetFrame(idleSuspend);
    defineComboBoxWidgetFrame(closeLidAction);

    // DRS dynamic resource scheduling
    QFrame *m_generalPowerDRSFrame;
    QHBoxLayout *m_DRSHBoxLayout;
    QLabel *m_generalPowerDRSLabel;
    FixLabel *m_DRSDetialLabel;
    KSwitchButton *m_generalPowerDRSBtn;
    void generalPowerDRSSettingsFrame(QWidget *widget);

    QSpacerItem *m_spacerItem;

    defineMainWidgetFrame(powerPlan);
    defineComboBoxWidgetFrame(acPolicy);
    defineComboBoxWidgetFrame(batteryPolicy);

    defineMainWidgetFrame(batterySavePlan);
    defineButtonWidgetFrame(batterySaveSwitch);
    defineButtonWidgetFrame(lowPowerAutoBatterySaveSwitch);
    defineButtonWidgetFrame(reduceBrightnessSwitch);
    defineComboBoxWidgetFrame(lowPowerNotification);

    defineComboBoxWidgetFrame(criticalPercentageAction);
    QLabel *m_criticalPercentageLable;
    QComboBox *m_criticalPercentageComboBox;
    QStringList m_criticalPercentageOptions;
    void criticalPercentageActionFrame();

    defineButtonWidgetFrame(showBatteryTime);

    void checkMachineType();
    void initLogin1DBus();
    void initUpowerDBus();
    void initPluginWidget(QWidget *widget);
    void setupLableText();
    void setupComboBoxText();
    void initWidgetValue();
    void connectWidgetSignals();

    void initMainFrame(
            QWidget *widget,
            QFrame **mainFrame,
            QVBoxLayout **vBoxLayout);
    void addButtonSubFrame(
            QFrame *fatherFrame, QFrame **subFrame,
            QVBoxLayout *vBoxLayout, QFrame **lineBetween,
            QLabel **subFrameLabel, KSwitchButton **button,
            const QString &buttonName);
    void addComboBoxSubFrame(
            QFrame *fatherFrame, QFrame **subFrame,
            QVBoxLayout *vBoxLayout, QFrame **lineBetween,
            QLabel **subFrameLabel, QComboBox **comboBox,
            const QString &comboBoxName);
    void setFrame_Noframe(QFrame *frame);
    bool QLabelSetText(QLabel *label, QString string);
    QFrame *setLine(QFrame *frame);

    bool buriedSettings(QString, QString, QString, QString);
    int getDeviceType(const QString &);

private slots:
    void setVisibleBySecurity();
    void dealUPMSettingsChanged(const QString &);
    void dealUPowerDBusMessage(QDBusMessage msg);
};

#endif // POWER_H
