#[macro_use]
extern crate pretty_assertions;

extern crate nom;
extern crate tls_parser;

mod tls_handshake {
    use nom::{Err, Needed};
    use tls_parser::*;

    #[rustfmt::skip]
static CH : &'static [u8] = &[
    0x16, 0x03, 0x01, 0x01, 0x2c, 0x01, 0x00, 0x01, 0x28, 0x03, 0x03, 0xb2,
    0x9d, 0xd7, 0x87, 0xff, 0x21, 0xeb, 0x04, 0xc8, 0xa5, 0x38, 0x39, 0x9a,
    0xcf, 0xb7, 0xa3, 0x82, 0x1f, 0x82, 0x6c, 0x49, 0xbc, 0x8b, 0xb8, 0xa9,
    0x03, 0x0a, 0x2d, 0xce, 0x38, 0x0b, 0xf4, 0x00, 0x00, 0xaa, 0xc0, 0x30,
    0xc0, 0x2c, 0xc0, 0x28, 0xc0, 0x24, 0xc0, 0x14, 0xc0, 0x0a, 0x00, 0xa5,
    0x00, 0xa3, 0x00, 0xa1, 0x00, 0x9f, 0x00, 0x6b, 0x00, 0x6a, 0x00, 0x69,
    0x00, 0x68, 0x00, 0x39, 0x00, 0x38, 0x00, 0x37, 0x00, 0x36, 0x00, 0x88,
    0x00, 0x87, 0x00, 0x86, 0x00, 0x85, 0xc0, 0x32, 0xc0, 0x2e, 0xc0, 0x2a,
    0xc0, 0x26, 0xc0, 0x0f, 0xc0, 0x05, 0x00, 0x9d, 0x00, 0x3d, 0x00, 0x35,
    0x00, 0x84, 0xc0, 0x2f, 0xc0, 0x2b, 0xc0, 0x27, 0xc0, 0x23, 0xc0, 0x13,
    0xc0, 0x09, 0x00, 0xa4, 0x00, 0xa2, 0x00, 0xa0, 0x00, 0x9e, 0x00, 0x67,
    0x00, 0x40, 0x00, 0x3f, 0x00, 0x3e, 0x00, 0x33, 0x00, 0x32, 0x00, 0x31,
    0x00, 0x30, 0x00, 0x9a, 0x00, 0x99, 0x00, 0x98, 0x00, 0x97, 0x00, 0x45,
    0x00, 0x44, 0x00, 0x43, 0x00, 0x42, 0xc0, 0x31, 0xc0, 0x2d, 0xc0, 0x29,
    0xc0, 0x25, 0xc0, 0x0e, 0xc0, 0x04, 0x00, 0x9c, 0x00, 0x3c, 0x00, 0x2f,
    0x00, 0x96, 0x00, 0x41, 0xc0, 0x11, 0xc0, 0x07, 0xc0, 0x0c, 0xc0, 0x02,
    0x00, 0x05, 0x00, 0x04, 0xc0, 0x12, 0xc0, 0x08, 0x00, 0x16, 0x00, 0x13,
    0x00, 0x10, 0x00, 0x0d, 0xc0, 0x0d, 0xc0, 0x03, 0x00, 0x0a, 0x00, 0xff,
    0x01, 0x00, 0x00, 0x55, 0x00, 0x0b, 0x00, 0x04, 0x03, 0x00, 0x01, 0x02,
    0x00, 0x0a, 0x00, 0x1c, 0x00, 0x1a, 0x00, 0x17, 0x00, 0x19, 0x00, 0x1c,
    0x00, 0x1b, 0x00, 0x18, 0x00, 0x1a, 0x00, 0x16, 0x00, 0x0e, 0x00, 0x0d,
    0x00, 0x0b, 0x00, 0x0c, 0x00, 0x09, 0x00, 0x0a, 0x00, 0x23, 0x00, 0x00,
    0x00, 0x0d, 0x00, 0x20, 0x00, 0x1e, 0x06, 0x01, 0x06, 0x02, 0x06, 0x03,
    0x05, 0x01, 0x05, 0x02, 0x05, 0x03, 0x04, 0x01, 0x04, 0x02, 0x04, 0x03,
    0x03, 0x01, 0x03, 0x02, 0x03, 0x03, 0x02, 0x01, 0x02, 0x02, 0x02, 0x03,
    0x00, 0x0f, 0x00, 0x01, 0x01
];

    #[test]
    fn test_tls_record_clienthello() {
        let empty = &b""[..];
        let rand_data = [
            0xff, 0x21, 0xeb, 0x04, 0xc8, 0xa5, 0x38, 0x39, 0x9a, 0xcf, 0xb7, 0xa3, 0x82, 0x1f,
            0x82, 0x6c, 0x49, 0xbc, 0x8b, 0xb8, 0xa9, 0x03, 0x0a, 0x2d, 0xce, 0x38, 0x0b, 0xf4,
        ];
        let ciphers = vec![
            0xc030, 0xc02c, 0xc028, 0xc024, 0xc014, 0xc00a, 0x00a5, 0x00a3, 0x00a1, 0x009f, 0x006b,
            0x006a, 0x0069, 0x0068, 0x0039, 0x0038, 0x0037, 0x0036, 0x0088, 0x0087, 0x0086, 0x0085,
            0xc032, 0xc02e, 0xc02a, 0xc026, 0xc00f, 0xc005, 0x009d, 0x003d, 0x0035, 0x0084, 0xc02f,
            0xc02b, 0xc027, 0xc023, 0xc013, 0xc009, 0x00a4, 0x00a2, 0x00a0, 0x009e, 0x0067, 0x0040,
            0x003f, 0x003e, 0x0033, 0x0032, 0x0031, 0x0030, 0x009a, 0x0099, 0x0098, 0x0097, 0x0045,
            0x0044, 0x0043, 0x0042, 0xc031, 0xc02d, 0xc029, 0xc025, 0xc00e, 0xc004, 0x009c, 0x003c,
            0x002f, 0x0096, 0x0041, 0xc011, 0xc007, 0xc00c, 0xc002, 0x0005, 0x0004, 0xc012, 0xc008,
            0x0016, 0x0013, 0x0010, 0x000d, 0xc00d, 0xc003, 0x000a, 0x00ff,
        ];
        let comp = vec![TlsCompressionID(0x00)];
        let expected = TlsPlaintext {
            hdr: TlsRecordHeader {
                record_type: TlsRecordType::Handshake,
                version: TlsVersion::Tls10,
                len: 300,
            },
            msg: vec![TlsMessage::Handshake(TlsMessageHandshake::ClientHello(
                TlsClientHelloContents {
                    version: TlsVersion::Tls12,
                    rand_time: 0xb29dd787,
                    rand_data: &rand_data,
                    session_id: None,
                    ciphers: ciphers.iter().map(|&x| TlsCipherSuiteID(x)).collect(),
                    comp: comp,
                    ext: Some(&CH[220..]),
                },
            ))],
        };
        let res = parse_tls_plaintext(CH);
        assert_eq!(res, Ok((empty, expected)));
    }

    // tls response, composed of 4 records:
    // - Server Hello
    // - Server Certificate
    // - Server Key Exchange
    // - Server Hello Done
    #[rustfmt::skip]
static SERVER_REPLY1: &'static [u8] = &[
    0x16, 0x03, 0x03, 0x00, 0x3b, 0x02, 0x00, 0x00, 0x37, 0x03, 0x03, 0x57,
    0xc4, 0x57, 0xda, 0x9c, 0xd3, 0x24, 0x6d, 0x9d, 0x02, 0x26, 0xa2, 0xe5,
    0x9a, 0xe8, 0xa5, 0x6f, 0x40, 0xad, 0x94, 0x30, 0xba, 0x49, 0x05, 0x3a,
    0x1e, 0x1b, 0xe1, 0x94, 0xa1, 0xba, 0x41, 0x00, 0xc0, 0x2f, 0x00, 0x00,
    0x0f, 0xff, 0x01, 0x00, 0x01, 0x00, 0x00, 0x23, 0x00, 0x00, 0x00, 0x0b,
    0x00, 0x02, 0x01, 0x00, 0x16, 0x03, 0x03, 0x0c, 0x09, 0x0b, 0x00, 0x0c,
    0x05, 0x00, 0x0c, 0x02, 0x00, 0x04, 0x84, 0x30, 0x82, 0x04, 0x80, 0x30,
    0x82, 0x03, 0x68, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x52, 0x1a,
    0x61, 0xda, 0x68, 0xb6, 0xe6, 0x35, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86,
    0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x30, 0x49, 0x31,
    0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53,
    0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13, 0x0a, 0x47,
    0x6f, 0x6f, 0x67, 0x6c, 0x65, 0x20, 0x49, 0x6e, 0x63, 0x31, 0x25, 0x30,
    0x23, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x1c, 0x47, 0x6f, 0x6f, 0x67,
    0x6c, 0x65, 0x20, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x6e, 0x65, 0x74, 0x20,
    0x41, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x69, 0x74, 0x79, 0x20, 0x47, 0x32,
    0x30, 0x1e, 0x17, 0x0d, 0x31, 0x36, 0x30, 0x38, 0x31, 0x37, 0x31, 0x38,
    0x34, 0x39, 0x30, 0x31, 0x5a, 0x17, 0x0d, 0x31, 0x36, 0x31, 0x31, 0x30,
    0x39, 0x31, 0x38, 0x32, 0x39, 0x30, 0x30, 0x5a, 0x30, 0x68, 0x31, 0x0b,
    0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31,
    0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x0a, 0x43, 0x61,
    0x6c, 0x69, 0x66, 0x6f, 0x72, 0x6e, 0x69, 0x61, 0x31, 0x16, 0x30, 0x14,
    0x06, 0x03, 0x55, 0x04, 0x07, 0x0c, 0x0d, 0x4d, 0x6f, 0x75, 0x6e, 0x74,
    0x61, 0x69, 0x6e, 0x20, 0x56, 0x69, 0x65, 0x77, 0x31, 0x13, 0x30, 0x11,
    0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c, 0x0a, 0x47, 0x6f, 0x6f, 0x67, 0x6c,
    0x65, 0x20, 0x49, 0x6e, 0x63, 0x31, 0x17, 0x30, 0x15, 0x06, 0x03, 0x55,
    0x04, 0x03, 0x0c, 0x0e, 0x77, 0x77, 0x77, 0x2e, 0x67, 0x6f, 0x6f, 0x67,
    0x6c, 0x65, 0x2e, 0x63, 0x6f, 0x6d, 0x30, 0x82, 0x01, 0x22, 0x30, 0x0d,
    0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05,
    0x00, 0x03, 0x82, 0x01, 0x0f, 0x00, 0x30, 0x82, 0x01, 0x0a, 0x02, 0x82,
    0x01, 0x01, 0x00, 0x9b, 0x5b, 0x36, 0xcc, 0xce, 0x8c, 0x39, 0x86, 0xbd,
    0x05, 0x2f, 0xc3, 0xff, 0x4c, 0x59, 0xa8, 0xfc, 0xe4, 0x6e, 0x0f, 0xc6,
    0xbf, 0x23, 0xf6, 0x35, 0xc5, 0x3b, 0xcc, 0xc9, 0x61, 0x2f, 0x60, 0x11,
    0x9b, 0x7a, 0x44, 0xd2, 0xd0, 0xd9, 0x62, 0x11, 0x09, 0xbe, 0x08, 0x53,
    0x24, 0x73, 0xd9, 0x5f, 0x5e, 0xb0, 0x6f, 0x05, 0x3f, 0x53, 0xa3, 0x2d,
    0x98, 0xa4, 0xf4, 0xaf, 0x23, 0x25, 0xb8, 0x80, 0xbc, 0xcc, 0xde, 0xa2,
    0x56, 0xcb, 0x6f, 0x9d, 0x1a, 0xda, 0xa5, 0xbe, 0xb0, 0x8a, 0xdc, 0xcf,
    0x30, 0xef, 0xe0, 0x68, 0x97, 0xc5, 0x70, 0xad, 0xcf, 0x10, 0x5b, 0xb6,
    0x92, 0x98, 0x61, 0x57, 0x3d, 0xbf, 0x3f, 0xd0, 0x5d, 0xbc, 0x76, 0x23,
    0xc8, 0xce, 0x35, 0x6d, 0x25, 0x1f, 0x11, 0x5a, 0x4b, 0x76, 0x36, 0xca,
    0xd4, 0x57, 0x7e, 0x95, 0x7f, 0x78, 0xfb, 0x2c, 0xe0, 0x9a, 0xbf, 0x64,
    0xc6, 0x43, 0x7b, 0xcd, 0xfa, 0x0e, 0xaa, 0x4f, 0x6a, 0xda, 0x04, 0xa4,
    0xf5, 0xfa, 0x2d, 0xea, 0xac, 0xc3, 0xa7, 0xf4, 0xd3, 0x7c, 0x57, 0xd4,
    0xb8, 0x2c, 0xcb, 0xe0, 0xd5, 0x26, 0x3b, 0x2d, 0x64, 0x0d, 0x20, 0x5a,
    0xd5, 0xe4, 0x1a, 0x6e, 0x3a, 0x6a, 0x23, 0xb1, 0x0a, 0xc6, 0x33, 0xee,
    0x49, 0x66, 0x13, 0x38, 0x58, 0x76, 0x53, 0x84, 0x4a, 0x8a, 0xaa, 0x77,
    0x8e, 0xa8, 0x38, 0xfb, 0xe9, 0x4d, 0xd0, 0x02, 0xf9, 0x7b, 0xf3, 0x67,
    0x94, 0xe0, 0x58, 0x9e, 0x04, 0xc1, 0x3b, 0xd8, 0x2b, 0x89, 0x5f, 0x57,
    0xf3, 0xb5, 0xf3, 0x41, 0x63, 0x7a, 0x32, 0x98, 0x5e, 0x18, 0x29, 0x04,
    0xb7, 0x64, 0xbe, 0x91, 0xf9, 0x15, 0xf7, 0x34, 0x23, 0x0f, 0x4d, 0xd1,
    0xeb, 0x80, 0xe9, 0xad, 0xb3, 0x65, 0xc1, 0xcd, 0x7e, 0x68, 0x3f, 0x9b,
    0x0c, 0x6d, 0xb2, 0xbf, 0x16, 0x11, 0x21, 0x02, 0x03, 0x01, 0x00, 0x01,
    0xa3, 0x82, 0x01, 0x4b, 0x30, 0x82, 0x01, 0x47, 0x30, 0x1d, 0x06, 0x03,
    0x55, 0x1d, 0x25, 0x04, 0x16, 0x30, 0x14, 0x06, 0x08, 0x2b, 0x06, 0x01,
    0x05, 0x05, 0x07, 0x03, 0x01, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05,
    0x07, 0x03, 0x02, 0x30, 0x19, 0x06, 0x03, 0x55, 0x1d, 0x11, 0x04, 0x12,
    0x30, 0x10, 0x82, 0x0e, 0x77, 0x77, 0x77, 0x2e, 0x67, 0x6f, 0x6f, 0x67,
    0x6c, 0x65, 0x2e, 0x63, 0x6f, 0x6d, 0x30, 0x68, 0x06, 0x08, 0x2b, 0x06,
    0x01, 0x05, 0x05, 0x07, 0x01, 0x01, 0x04, 0x5c, 0x30, 0x5a, 0x30, 0x2b,
    0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x30, 0x02, 0x86, 0x1f,
    0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x70, 0x6b, 0x69, 0x2e, 0x67,
    0x6f, 0x6f, 0x67, 0x6c, 0x65, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x47, 0x49,
    0x41, 0x47, 0x32, 0x2e, 0x63, 0x72, 0x74, 0x30, 0x2b, 0x06, 0x08, 0x2b,
    0x06, 0x01, 0x05, 0x05, 0x07, 0x30, 0x01, 0x86, 0x1f, 0x68, 0x74, 0x74,
    0x70, 0x3a, 0x2f, 0x2f, 0x63, 0x6c, 0x69, 0x65, 0x6e, 0x74, 0x73, 0x31,
    0x2e, 0x67, 0x6f, 0x6f, 0x67, 0x6c, 0x65, 0x2e, 0x63, 0x6f, 0x6d, 0x2f,
    0x6f, 0x63, 0x73, 0x70, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04,
    0x16, 0x04, 0x14, 0x2e, 0x67, 0x92, 0x22, 0x0f, 0xd3, 0xd5, 0xd1, 0xd6,
    0x0a, 0x21, 0xf5, 0x3e, 0x2b, 0xe7, 0x87, 0xbe, 0xaf, 0x37, 0x9b, 0x30,
    0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30,
    0x00, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16,
    0x80, 0x14, 0x4a, 0xdd, 0x06, 0x16, 0x1b, 0xbc, 0xf6, 0x68, 0xb5, 0x76,
    0xf5, 0x81, 0xb6, 0xbb, 0x62, 0x1a, 0xba, 0x5a, 0x81, 0x2f, 0x30, 0x21,
    0x06, 0x03, 0x55, 0x1d, 0x20, 0x04, 0x1a, 0x30, 0x18, 0x30, 0x0c, 0x06,
    0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0xd6, 0x79, 0x02, 0x05, 0x01, 0x30,
    0x08, 0x06, 0x06, 0x67, 0x81, 0x0c, 0x01, 0x02, 0x02, 0x30, 0x30, 0x06,
    0x03, 0x55, 0x1d, 0x1f, 0x04, 0x29, 0x30, 0x27, 0x30, 0x25, 0xa0, 0x23,
    0xa0, 0x21, 0x86, 0x1f, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x70,
    0x6b, 0x69, 0x2e, 0x67, 0x6f, 0x6f, 0x67, 0x6c, 0x65, 0x2e, 0x63, 0x6f,
    0x6d, 0x2f, 0x47, 0x49, 0x41, 0x47, 0x32, 0x2e, 0x63, 0x72, 0x6c, 0x30,
    0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b,
    0x05, 0x00, 0x03, 0x82, 0x01, 0x01, 0x00, 0x7d, 0x0e, 0x9e, 0x7b, 0xf0,
    0x81, 0xd6, 0x19, 0xa5, 0xc1, 0xe3, 0xd4, 0x16, 0xad, 0x17, 0x74, 0x76,
    0x80, 0x97, 0x9c, 0x96, 0x7b, 0xfb, 0x8e, 0x3a, 0x12, 0x14, 0x25, 0x48,
    0xb9, 0xed, 0x46, 0xc9, 0x07, 0xc6, 0xd9, 0xfd, 0x06, 0xba, 0x9f, 0x66,
    0xd6, 0x1c, 0xe4, 0xbb, 0xcb, 0x76, 0x44, 0x95, 0x31, 0x76, 0x4c, 0xfb,
    0xb5, 0xad, 0xc7, 0xf0, 0x6a, 0xfa, 0x30, 0xc3, 0x58, 0xa1, 0x8b, 0xcc,
    0xe2, 0x70, 0xbb, 0x8a, 0x78, 0x5e, 0x41, 0x60, 0x1e, 0xda, 0x69, 0xc3,
    0xcb, 0x9f, 0xa2, 0x10, 0xff, 0x20, 0xfc, 0xa4, 0x35, 0xdb, 0x32, 0xcb,
    0x26, 0xba, 0xc0, 0xcf, 0x3f, 0x5c, 0x86, 0x78, 0x2c, 0xaa, 0x73, 0x39,
    0xf9, 0x8c, 0xfd, 0xa6, 0x9a, 0x8e, 0x3f, 0x5d, 0x7d, 0x8c, 0x7c, 0x3a,
    0xe7, 0xbe, 0x69, 0x73, 0xa8, 0x06, 0x5d, 0x56, 0xe3, 0xe2, 0x80, 0x0d,
    0x99, 0xcc, 0x4d, 0x5f, 0x7e, 0xfb, 0x62, 0xd8, 0xcf, 0x55, 0x2d, 0xbc,
    0x9f, 0xef, 0xa7, 0xe6, 0x46, 0xda, 0xc9, 0x66, 0x07, 0x0e, 0x13, 0x4d,
    0x61, 0xb3, 0x85, 0xf3, 0x72, 0x6d, 0x41, 0xfe, 0x7b, 0xaa, 0xe7, 0xaf,
    0xff, 0x61, 0xd9, 0xc6, 0x6c, 0x84, 0x6c, 0x1a, 0x9e, 0xda, 0x86, 0x15,
    0xbc, 0x2a, 0xae, 0x84, 0x14, 0x94, 0x67, 0x95, 0xc4, 0xda, 0x35, 0x29,
    0x8e, 0x80, 0x23, 0xc0, 0x4a, 0xdd, 0x0e, 0xb3, 0xb1, 0x7d, 0xc7, 0x3d,
    0xea, 0x2c, 0x41, 0xb2, 0xed, 0xb4, 0x8b, 0x65, 0xd3, 0x62, 0x81, 0x0a,
    0x25, 0x0c, 0x3b, 0xc9, 0x11, 0xa2, 0x50, 0x1f, 0xd0, 0xe9, 0x37, 0x9f,
    0xa0, 0x7a, 0x34, 0x01, 0xdf, 0x32, 0x58, 0x3b, 0xf5, 0x4b, 0xd8, 0xab,
    0x11, 0x5a, 0x9a, 0x1b, 0x83, 0x43, 0x06, 0xf4, 0x40, 0x1e, 0x5b, 0x92,
    0xe8, 0x94, 0x0a, 0xbc, 0x57, 0x2f, 0x19, 0x69, 0xc6, 0xd2, 0x8b, 0x00,
    0x03, 0xf4, 0x30, 0x82, 0x03, 0xf0, 0x30, 0x82, 0x02, 0xd8, 0xa0, 0x03,
    0x02, 0x01, 0x02, 0x02, 0x03, 0x02, 0x3a, 0x92, 0x30, 0x0d, 0x06, 0x09,
    0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x30,
    0x42, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02,
    0x55, 0x53, 0x31, 0x16, 0x30, 0x14, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13,
    0x0d, 0x47, 0x65, 0x6f, 0x54, 0x72, 0x75, 0x73, 0x74, 0x20, 0x49, 0x6e,
    0x63, 0x2e, 0x31, 0x1b, 0x30, 0x19, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13,
    0x12, 0x47, 0x65, 0x6f, 0x54, 0x72, 0x75, 0x73, 0x74, 0x20, 0x47, 0x6c,
    0x6f, 0x62, 0x61, 0x6c, 0x20, 0x43, 0x41, 0x30, 0x1e, 0x17, 0x0d, 0x31,
    0x35, 0x30, 0x34, 0x30, 0x31, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5a,
    0x17, 0x0d, 0x31, 0x37, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39,
    0x35, 0x39, 0x5a, 0x30, 0x49, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55,
    0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03,
    0x55, 0x04, 0x0a, 0x13, 0x0a, 0x47, 0x6f, 0x6f, 0x67, 0x6c, 0x65, 0x20,
    0x49, 0x6e, 0x63, 0x31, 0x25, 0x30, 0x23, 0x06, 0x03, 0x55, 0x04, 0x03,
    0x13, 0x1c, 0x47, 0x6f, 0x6f, 0x67, 0x6c, 0x65, 0x20, 0x49, 0x6e, 0x74,
    0x65, 0x72, 0x6e, 0x65, 0x74, 0x20, 0x41, 0x75, 0x74, 0x68, 0x6f, 0x72,
    0x69, 0x74, 0x79, 0x20, 0x47, 0x32, 0x30, 0x82, 0x01, 0x22, 0x30, 0x0d,
    0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05,
    0x00, 0x03, 0x82, 0x01, 0x0f, 0x00, 0x30, 0x82, 0x01, 0x0a, 0x02, 0x82,
    0x01, 0x01, 0x00, 0x9c, 0x2a, 0x04, 0x77, 0x5c, 0xd8, 0x50, 0x91, 0x3a,
    0x06, 0xa3, 0x82, 0xe0, 0xd8, 0x50, 0x48, 0xbc, 0x89, 0x3f, 0xf1, 0x19,
    0x70, 0x1a, 0x88, 0x46, 0x7e, 0xe0, 0x8f, 0xc5, 0xf1, 0x89, 0xce, 0x21,
    0xee, 0x5a, 0xfe, 0x61, 0x0d, 0xb7, 0x32, 0x44, 0x89, 0xa0, 0x74, 0x0b,
    0x53, 0x4f, 0x55, 0xa4, 0xce, 0x82, 0x62, 0x95, 0xee, 0xeb, 0x59, 0x5f,
    0xc6, 0xe1, 0x05, 0x80, 0x12, 0xc4, 0x5e, 0x94, 0x3f, 0xbc, 0x5b, 0x48,
    0x38, 0xf4, 0x53, 0xf7, 0x24, 0xe6, 0xfb, 0x91, 0xe9, 0x15, 0xc4, 0xcf,
    0xf4, 0x53, 0x0d, 0xf4, 0x4a, 0xfc, 0x9f, 0x54, 0xde, 0x7d, 0xbe, 0xa0,
    0x6b, 0x6f, 0x87, 0xc0, 0xd0, 0x50, 0x1f, 0x28, 0x30, 0x03, 0x40, 0xda,
    0x08, 0x73, 0x51, 0x6c, 0x7f, 0xff, 0x3a, 0x3c, 0xa7, 0x37, 0x06, 0x8e,
    0xbd, 0x4b, 0x11, 0x04, 0xeb, 0x7d, 0x24, 0xde, 0xe6, 0xf9, 0xfc, 0x31,
    0x71, 0xfb, 0x94, 0xd5, 0x60, 0xf3, 0x2e, 0x4a, 0xaf, 0x42, 0xd2, 0xcb,
    0xea, 0xc4, 0x6a, 0x1a, 0xb2, 0xcc, 0x53, 0xdd, 0x15, 0x4b, 0x8b, 0x1f,
    0xc8, 0x19, 0x61, 0x1f, 0xcd, 0x9d, 0xa8, 0x3e, 0x63, 0x2b, 0x84, 0x35,
    0x69, 0x65, 0x84, 0xc8, 0x19, 0xc5, 0x46, 0x22, 0xf8, 0x53, 0x95, 0xbe,
    0xe3, 0x80, 0x4a, 0x10, 0xc6, 0x2a, 0xec, 0xba, 0x97, 0x20, 0x11, 0xc7,
    0x39, 0x99, 0x10, 0x04, 0xa0, 0xf0, 0x61, 0x7a, 0x95, 0x25, 0x8c, 0x4e,
    0x52, 0x75, 0xe2, 0xb6, 0xed, 0x08, 0xca, 0x14, 0xfc, 0xce, 0x22, 0x6a,
    0xb3, 0x4e, 0xcf, 0x46, 0x03, 0x97, 0x97, 0x03, 0x7e, 0xc0, 0xb1, 0xde,
    0x7b, 0xaf, 0x45, 0x33, 0xcf, 0xba, 0x3e, 0x71, 0xb7, 0xde, 0xf4, 0x25,
    0x25, 0xc2, 0x0d, 0x35, 0x89, 0x9d, 0x9d, 0xfb, 0x0e, 0x11, 0x79, 0x89,
    0x1e, 0x37, 0xc5, 0xaf, 0x8e, 0x72, 0x69, 0x02, 0x03, 0x01, 0x00, 0x01,
    0xa3, 0x81, 0xe7, 0x30, 0x81, 0xe4, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d,
    0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0xc0, 0x7a, 0x98, 0x68, 0x8d,
    0x89, 0xfb, 0xab, 0x05, 0x64, 0x0c, 0x11, 0x7d, 0xaa, 0x7d, 0x65, 0xb8,
    0xca, 0xcc, 0x4e, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16,
    0x04, 0x14, 0x4a, 0xdd, 0x06, 0x16, 0x1b, 0xbc, 0xf6, 0x68, 0xb5, 0x76,
    0xf5, 0x81, 0xb6, 0xbb, 0x62, 0x1a, 0xba, 0x5a, 0x81, 0x2f, 0x30, 0x0e,
    0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04, 0x03, 0x02,
    0x01, 0x06, 0x30, 0x2e, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07,
    0x01, 0x01, 0x04, 0x22, 0x30, 0x20, 0x30, 0x1e, 0x06, 0x08, 0x2b, 0x06,
    0x01, 0x05, 0x05, 0x07, 0x30, 0x01, 0x86, 0x12, 0x68, 0x74, 0x74, 0x70,
    0x3a, 0x2f, 0x2f, 0x67, 0x2e, 0x73, 0x79, 0x6d, 0x63, 0x64, 0x2e, 0x63,
    0x6f, 0x6d, 0x30, 0x12, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff,
    0x04, 0x08, 0x30, 0x06, 0x01, 0x01, 0xff, 0x02, 0x01, 0x00, 0x30, 0x35,
    0x06, 0x03, 0x55, 0x1d, 0x1f, 0x04, 0x2e, 0x30, 0x2c, 0x30, 0x2a, 0xa0,
    0x28, 0xa0, 0x26, 0x86, 0x24, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f,
    0x67, 0x2e, 0x73, 0x79, 0x6d, 0x63, 0x62, 0x2e, 0x63, 0x6f, 0x6d, 0x2f,
    0x63, 0x72, 0x6c, 0x73, 0x2f, 0x67, 0x74, 0x67, 0x6c, 0x6f, 0x62, 0x61,
    0x6c, 0x2e, 0x63, 0x72, 0x6c, 0x30, 0x17, 0x06, 0x03, 0x55, 0x1d, 0x20,
    0x04, 0x10, 0x30, 0x0e, 0x30, 0x0c, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04,
    0x01, 0xd6, 0x79, 0x02, 0x05, 0x01, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86,
    0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x03, 0x82, 0x01,
    0x01, 0x00, 0x08, 0x4e, 0x04, 0xa7, 0x80, 0x7f, 0x10, 0x16, 0x43, 0x5e,
    0x02, 0xad, 0xd7, 0x42, 0x80, 0xf4, 0xb0, 0x8e, 0xd2, 0xae, 0xb3, 0xeb,
    0x11, 0x7d, 0x90, 0x84, 0x18, 0x7d, 0xe7, 0x90, 0x15, 0xfb, 0x49, 0x7f,
    0xa8, 0x99, 0x05, 0x91, 0xbb, 0x7a, 0xc9, 0xd6, 0x3c, 0x37, 0x18, 0x09,
    0x9a, 0xb6, 0xc7, 0x92, 0x20, 0x07, 0x35, 0x33, 0x09, 0xe4, 0x28, 0x63,
    0x72, 0x0d, 0xb4, 0xe0, 0x32, 0x9c, 0x87, 0x98, 0xc4, 0x1b, 0x76, 0x89,
    0x67, 0xc1, 0x50, 0x58, 0xb0, 0x13, 0xaa, 0x13, 0x1a, 0x1b, 0x32, 0xa5,
    0xbe, 0xea, 0x11, 0x95, 0x4c, 0x48, 0x63, 0x49, 0xe9, 0x99, 0x5d, 0x20,
    0x37, 0xcc, 0xfe, 0x2a, 0x69, 0x51, 0x16, 0x95, 0x4b, 0xa9, 0xde, 0x49,
    0x82, 0xc0, 0x10, 0x70, 0xf4, 0x2c, 0xf3, 0xec, 0xbc, 0x24, 0x24, 0xd0,
    0x4e, 0xac, 0xa5, 0xd9, 0x5e, 0x1e, 0x6d, 0x92, 0xc1, 0xa7, 0xac, 0x48,
    0x35, 0x81, 0xf9, 0xe5, 0xe4, 0x9c, 0x65, 0x69, 0xcd, 0x87, 0xa4, 0x41,
    0x50, 0x3f, 0x2e, 0x57, 0xa5, 0x91, 0x51, 0x12, 0x58, 0x0e, 0x8c, 0x09,
    0xa1, 0xac, 0x7a, 0xa4, 0x12, 0xa5, 0x27, 0xf3, 0x9a, 0x10, 0x97, 0x7d,
    0x55, 0x03, 0x06, 0xf7, 0x66, 0x58, 0x5f, 0x5f, 0x64, 0xe1, 0xab, 0x5d,
    0x6d, 0xa5, 0x39, 0x48, 0x75, 0x98, 0x4c, 0x29, 0x5a, 0x3a, 0x8d, 0xd3,
    0x2b, 0xca, 0x9c, 0x55, 0x04, 0xbf, 0xf4, 0xe6, 0x14, 0xd5, 0x80, 0xac,
    0x26, 0xed, 0x17, 0x89, 0xa6, 0x93, 0x6c, 0x5c, 0xa4, 0xcc, 0xb8, 0xf0,
    0x66, 0x8e, 0x64, 0xe3, 0x7d, 0x9a, 0xe2, 0x00, 0xb3, 0x49, 0xc7, 0xe4,
    0x0a, 0xaa, 0xdd, 0x5b, 0x83, 0xc7, 0x70, 0x90, 0x46, 0x4e, 0xbe, 0xd0,
    0xdb, 0x59, 0x96, 0x6c, 0x2e, 0xf5, 0x16, 0x36, 0xde, 0x71, 0xcc, 0x01,
    0xc2, 0x12, 0xc1, 0x21, 0xc6, 0x16, 0x00, 0x03, 0x81, 0x30, 0x82, 0x03,
    0x7d, 0x30, 0x82, 0x02, 0xe6, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x03,
    0x12, 0xbb, 0xe6, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
    0x0d, 0x01, 0x01, 0x05, 0x05, 0x00, 0x30, 0x4e, 0x31, 0x0b, 0x30, 0x09,
    0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x10, 0x30,
    0x0e, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13, 0x07, 0x45, 0x71, 0x75, 0x69,
    0x66, 0x61, 0x78, 0x31, 0x2d, 0x30, 0x2b, 0x06, 0x03, 0x55, 0x04, 0x0b,
    0x13, 0x24, 0x45, 0x71, 0x75, 0x69, 0x66, 0x61, 0x78, 0x20, 0x53, 0x65,
    0x63, 0x75, 0x72, 0x65, 0x20, 0x43, 0x65, 0x72, 0x74, 0x69, 0x66, 0x69,
    0x63, 0x61, 0x74, 0x65, 0x20, 0x41, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x69,
    0x74, 0x79, 0x30, 0x1e, 0x17, 0x0d, 0x30, 0x32, 0x30, 0x35, 0x32, 0x31,
    0x30, 0x34, 0x30, 0x30, 0x30, 0x30, 0x5a, 0x17, 0x0d, 0x31, 0x38, 0x30,
    0x38, 0x32, 0x31, 0x30, 0x34, 0x30, 0x30, 0x30, 0x30, 0x5a, 0x30, 0x42,
    0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55,
    0x53, 0x31, 0x16, 0x30, 0x14, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13, 0x0d,
    0x47, 0x65, 0x6f, 0x54, 0x72, 0x75, 0x73, 0x74, 0x20, 0x49, 0x6e, 0x63,
    0x2e, 0x31, 0x1b, 0x30, 0x19, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x12,
    0x47, 0x65, 0x6f, 0x54, 0x72, 0x75, 0x73, 0x74, 0x20, 0x47, 0x6c, 0x6f,
    0x62, 0x61, 0x6c, 0x20, 0x43, 0x41, 0x30, 0x82, 0x01, 0x22, 0x30, 0x0d,
    0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05,
    0x00, 0x03, 0x82, 0x01, 0x0f, 0x00, 0x30, 0x82, 0x01, 0x0a, 0x02, 0x82,
    0x01, 0x01, 0x00, 0xda, 0xcc, 0x18, 0x63, 0x30, 0xfd, 0xf4, 0x17, 0x23,
    0x1a, 0x56, 0x7e, 0x5b, 0xdf, 0x3c, 0x6c, 0x38, 0xe4, 0x71, 0xb7, 0x78,
    0x91, 0xd4, 0xbc, 0xa1, 0xd8, 0x4c, 0xf8, 0xa8, 0x43, 0xb6, 0x03, 0xe9,
    0x4d, 0x21, 0x07, 0x08, 0x88, 0xda, 0x58, 0x2f, 0x66, 0x39, 0x29, 0xbd,
    0x05, 0x78, 0x8b, 0x9d, 0x38, 0xe8, 0x05, 0xb7, 0x6a, 0x7e, 0x71, 0xa4,
    0xe6, 0xc4, 0x60, 0xa6, 0xb0, 0xef, 0x80, 0xe4, 0x89, 0x28, 0x0f, 0x9e,
    0x25, 0xd6, 0xed, 0x83, 0xf3, 0xad, 0xa6, 0x91, 0xc7, 0x98, 0xc9, 0x42,
    0x18, 0x35, 0x14, 0x9d, 0xad, 0x98, 0x46, 0x92, 0x2e, 0x4f, 0xca, 0xf1,
    0x87, 0x43, 0xc1, 0x16, 0x95, 0x57, 0x2d, 0x50, 0xef, 0x89, 0x2d, 0x80,
    0x7a, 0x57, 0xad, 0xf2, 0xee, 0x5f, 0x6b, 0xd2, 0x00, 0x8d, 0xb9, 0x14,
    0xf8, 0x14, 0x15, 0x35, 0xd9, 0xc0, 0x46, 0xa3, 0x7b, 0x72, 0xc8, 0x91,
    0xbf, 0xc9, 0x55, 0x2b, 0xcd, 0xd0, 0x97, 0x3e, 0x9c, 0x26, 0x64, 0xcc,
    0xdf, 0xce, 0x83, 0x19, 0x71, 0xca, 0x4e, 0xe6, 0xd4, 0xd5, 0x7b, 0xa9,
    0x19, 0xcd, 0x55, 0xde, 0xc8, 0xec, 0xd2, 0x5e, 0x38, 0x53, 0xe5, 0x5c,
    0x4f, 0x8c, 0x2d, 0xfe, 0x50, 0x23, 0x36, 0xfc, 0x66, 0xe6, 0xcb, 0x8e,
    0xa4, 0x39, 0x19, 0x00, 0xb7, 0x95, 0x02, 0x39, 0x91, 0x0b, 0x0e, 0xfe,
    0x38, 0x2e, 0xd1, 0x1d, 0x05, 0x9a, 0xf6, 0x4d, 0x3e, 0x6f, 0x0f, 0x07,
    0x1d, 0xaf, 0x2c, 0x1e, 0x8f, 0x60, 0x39, 0xe2, 0xfa, 0x36, 0x53, 0x13,
    0x39, 0xd4, 0x5e, 0x26, 0x2b, 0xdb, 0x3d, 0xa8, 0x14, 0xbd, 0x32, 0xeb,
    0x18, 0x03, 0x28, 0x52, 0x04, 0x71, 0xe5, 0xab, 0x33, 0x3d, 0xe1, 0x38,
    0xbb, 0x07, 0x36, 0x84, 0x62, 0x9c, 0x79, 0xea, 0x16, 0x30, 0xf4, 0x5f,
    0xc0, 0x2b, 0xe8, 0x71, 0x6b, 0xe4, 0xf9, 0x02, 0x03, 0x01, 0x00, 0x01,
    0xa3, 0x81, 0xf0, 0x30, 0x81, 0xed, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d,
    0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x48, 0xe6, 0x68, 0xf9, 0x2b,
    0xd2, 0xb2, 0x95, 0xd7, 0x47, 0xd8, 0x23, 0x20, 0x10, 0x4f, 0x33, 0x98,
    0x90, 0x9f, 0xd4, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16,
    0x04, 0x14, 0xc0, 0x7a, 0x98, 0x68, 0x8d, 0x89, 0xfb, 0xab, 0x05, 0x64,
    0x0c, 0x11, 0x7d, 0xaa, 0x7d, 0x65, 0xb8, 0xca, 0xcc, 0x4e, 0x30, 0x0f,
    0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x05, 0x30, 0x03,
    0x01, 0x01, 0xff, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01,
    0xff, 0x04, 0x04, 0x03, 0x02, 0x01, 0x06, 0x30, 0x3a, 0x06, 0x03, 0x55,
    0x1d, 0x1f, 0x04, 0x33, 0x30, 0x31, 0x30, 0x2f, 0xa0, 0x2d, 0xa0, 0x2b,
    0x86, 0x29, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x63, 0x72, 0x6c,
    0x2e, 0x67, 0x65, 0x6f, 0x74, 0x72, 0x75, 0x73, 0x74, 0x2e, 0x63, 0x6f,
    0x6d, 0x2f, 0x63, 0x72, 0x6c, 0x73, 0x2f, 0x73, 0x65, 0x63, 0x75, 0x72,
    0x65, 0x63, 0x61, 0x2e, 0x63, 0x72, 0x6c, 0x30, 0x4e, 0x06, 0x03, 0x55,
    0x1d, 0x20, 0x04, 0x47, 0x30, 0x45, 0x30, 0x43, 0x06, 0x04, 0x55, 0x1d,
    0x20, 0x00, 0x30, 0x3b, 0x30, 0x39, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05,
    0x05, 0x07, 0x02, 0x01, 0x16, 0x2d, 0x68, 0x74, 0x74, 0x70, 0x73, 0x3a,
    0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x67, 0x65, 0x6f, 0x74, 0x72, 0x75,
    0x73, 0x74, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x72, 0x65, 0x73, 0x6f, 0x75,
    0x72, 0x63, 0x65, 0x73, 0x2f, 0x72, 0x65, 0x70, 0x6f, 0x73, 0x69, 0x74,
    0x6f, 0x72, 0x79, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
    0x0d, 0x01, 0x01, 0x05, 0x05, 0x00, 0x03, 0x81, 0x81, 0x00, 0x76, 0xe1,
    0x12, 0x6e, 0x4e, 0x4b, 0x16, 0x12, 0x86, 0x30, 0x06, 0xb2, 0x81, 0x08,
    0xcf, 0xf0, 0x08, 0xc7, 0xc7, 0x71, 0x7e, 0x66, 0xee, 0xc2, 0xed, 0xd4,
    0x3b, 0x1f, 0xff, 0xf0, 0xf0, 0xc8, 0x4e, 0xd6, 0x43, 0x38, 0xb0, 0xb9,
    0x30, 0x7d, 0x18, 0xd0, 0x55, 0x83, 0xa2, 0x6a, 0xcb, 0x36, 0x11, 0x9c,
    0xe8, 0x48, 0x66, 0xa3, 0x6d, 0x7f, 0xb8, 0x13, 0xd4, 0x47, 0xfe, 0x8b,
    0x5a, 0x5c, 0x73, 0xfc, 0xae, 0xd9, 0x1b, 0x32, 0x19, 0x38, 0xab, 0x97,
    0x34, 0x14, 0xaa, 0x96, 0xd2, 0xeb, 0xa3, 0x1c, 0x14, 0x08, 0x49, 0xb6,
    0xbb, 0xe5, 0x91, 0xef, 0x83, 0x36, 0xeb, 0x1d, 0x56, 0x6f, 0xca, 0xda,
    0xbc, 0x73, 0x63, 0x90, 0xe4, 0x7f, 0x7b, 0x3e, 0x22, 0xcb, 0x3d, 0x07,
    0xed, 0x5f, 0x38, 0x74, 0x9c, 0xe3, 0x03, 0x50, 0x4e, 0xa1, 0xaf, 0x98,
    0xee, 0x61, 0xf2, 0x84, 0x3f, 0x12, 0x16, 0x03, 0x03, 0x01, 0x4d, 0x0c,
    0x00, 0x01, 0x49, 0x03, 0x00, 0x17, 0x41, 0x04, 0x27, 0x7e, 0x6c, 0x81,
    0xf1, 0xb4, 0xb5, 0x60, 0x22, 0x45, 0x9b, 0x26, 0x4f, 0xb8, 0xa8, 0xbd,
    0xd2, 0x91, 0x88, 0x9b, 0xe0, 0xe8, 0x41, 0x4d, 0x7c, 0xab, 0xcf, 0xf7,
    0x3f, 0x8f, 0x23, 0xd1, 0x18, 0xdd, 0xb7, 0x27, 0xbd, 0xcf, 0xd3, 0x14,
    0x71, 0xca, 0xf9, 0x7e, 0xb8, 0xe8, 0x15, 0x59, 0xa2, 0x97, 0x66, 0x27,
    0x45, 0xc3, 0x40, 0xf7, 0xa8, 0x77, 0x61, 0x05, 0x9b, 0xb2, 0xa9, 0xd0,
    0x06, 0x01, 0x01, 0x00, 0x6e, 0xd0, 0x13, 0x15, 0x52, 0x29, 0xdb, 0x2c,
    0x60, 0x3c, 0x2d, 0xf3, 0x6f, 0xc5, 0xac, 0x44, 0x53, 0x47, 0x3b, 0x17,
    0xbc, 0xaf, 0xaa, 0x9f, 0x5d, 0xf5, 0x2e, 0x7a, 0x9c, 0x32, 0x58, 0xb5,
    0x37, 0x1a, 0xce, 0x4c, 0x12, 0x9e, 0x47, 0xf0, 0xfd, 0x85, 0x38, 0x39,
    0xc1, 0xec, 0xd8, 0x06, 0xc8, 0x84, 0x87, 0x03, 0xfa, 0x41, 0xf9, 0x88,
    0xa0, 0xef, 0x84, 0x87, 0x76, 0x6e, 0x61, 0xfc, 0x1c, 0x78, 0xd8, 0x70,
    0xa8, 0x65, 0x3a, 0x1e, 0x84, 0xac, 0x14, 0x78, 0x74, 0x18, 0x7f, 0xdd,
    0x35, 0x2e, 0x99, 0xd8, 0x3d, 0x68, 0x66, 0x16, 0x7a, 0x83, 0x16, 0xa6,
    0x21, 0x18, 0x71, 0x6f, 0x58, 0x32, 0x60, 0xa3, 0x70, 0xa6, 0xca, 0x04,
    0xd5, 0x09, 0xbe, 0xc3, 0x32, 0xc9, 0xee, 0x5d, 0x9a, 0x56, 0xdb, 0xf6,
    0x17, 0xf1, 0x2c, 0x6d, 0x71, 0x4f, 0xf7, 0x8a, 0x2a, 0xa3, 0xcf, 0xb9,
    0x86, 0x0a, 0xc2, 0xfd, 0x75, 0xab, 0xb2, 0x75, 0x39, 0xb5, 0xf5, 0x2b,
    0xb2, 0x80, 0x9e, 0x9b, 0x32, 0x26, 0x25, 0x6c, 0x0e, 0x71, 0xdf, 0xc0,
    0x42, 0x4e, 0x74, 0xd8, 0xb0, 0x9b, 0xa5, 0x15, 0xe5, 0x96, 0xd7, 0x30,
    0xdf, 0x33, 0x3d, 0xfd, 0xba, 0xf7, 0x59, 0x7d, 0xdb, 0xc9, 0x31, 0x3d,
    0x70, 0xe4, 0xd4, 0x91, 0x97, 0x70, 0x2d, 0xf2, 0x48, 0xcd, 0x84, 0x2d,
    0x70, 0x48, 0xbc, 0xd6, 0x6b, 0xaf, 0xdd, 0xf6, 0x7d, 0xad, 0xb9, 0x89,
    0x94, 0x7c, 0x59, 0x0c, 0x3f, 0x3e, 0xe2, 0x8d, 0x4c, 0x81, 0x4a, 0x15,
    0x09, 0xb9, 0xde, 0xc1, 0xe6, 0xe6, 0x5d, 0x28, 0x2d, 0x1d, 0xb8, 0x45,
    0x17, 0x42, 0x55, 0xe3, 0x2a, 0xcf, 0x55, 0x26, 0x66, 0x79, 0xf1, 0xbb,
    0x2a, 0x25, 0x28, 0x78, 0xa1, 0x63, 0x90, 0xec, 0xec, 0xa7, 0xee, 0x61,
    0x1c, 0xac, 0x19, 0x45, 0xdd, 0x82, 0xae, 0x52, 0x16, 0x03, 0x03, 0x00,
    0x04, 0x0e, 0x00, 0x00, 0x00
];

    #[test]
    fn test_tls_record_serverhello() {
        let empty = &b""[..];
        let bytes = &SERVER_REPLY1[0..64];
        let expected = TlsPlaintext {
            hdr: TlsRecordHeader {
                record_type: TlsRecordType::Handshake,
                version: TlsVersion::Tls12,
                len: 59,
            },
            msg: vec![TlsMessage::Handshake(TlsMessageHandshake::ServerHello(
                TlsServerHelloContents {
                    version: TlsVersion::Tls12,
                    rand_time: 0x57c457da,
                    rand_data: &bytes[15..43],
                    session_id: None,
                    cipher: TlsCipherSuiteID(0xc02f),
                    compression: TlsCompressionID(0),
                    ext: Some(&bytes[49..]),
                },
            ))],
        };
        assert_eq!(parse_tls_plaintext(&bytes), Ok((empty, expected)));
    }

    #[test]
    fn test_tls_record_certificate() {
        let empty = &b""[..];
        let bytes = &SERVER_REPLY1[64..3150];
        let chain = vec![
            RawCertificate {
                data: &bytes[15..1171],
            },
            RawCertificate {
                data: &bytes[1174..2186],
            },
            RawCertificate {
                data: &bytes[2189..3086],
            },
        ];
        for cert in &chain {
            println!("cert len: {}", cert.data.len());
        }
        let expected = TlsPlaintext {
            hdr: TlsRecordHeader {
                record_type: TlsRecordType::Handshake,
                version: TlsVersion::Tls12,
                len: 3081,
            },
            msg: vec![TlsMessage::Handshake(TlsMessageHandshake::Certificate(
                TlsCertificateContents { cert_chain: chain },
            ))],
        };
        assert_eq!(parse_tls_plaintext(&bytes), Ok((empty, expected)));
    }

    #[test]
    fn test_tls_record_serverkeyexchange() {
        let empty = &b""[..];
        let bytes = &SERVER_REPLY1[3150..3488];
        let expected = TlsPlaintext {
            hdr: TlsRecordHeader {
                record_type: TlsRecordType::Handshake,
                version: TlsVersion::Tls12,
                len: 333,
            },
            msg: vec![TlsMessage::Handshake(
                TlsMessageHandshake::ServerKeyExchange(TlsServerKeyExchangeContents {
                    parameters: &bytes[9..],
                }),
            )],
        };
        assert_eq!(parse_tls_plaintext(&bytes), Ok((empty, expected)));
    }

    #[test]
    fn test_tls_record_serverdone() {
        let empty = &b""[..];
        let bytes = &SERVER_REPLY1[3488..];
        let expected = TlsPlaintext {
            hdr: TlsRecordHeader {
                record_type: TlsRecordType::Handshake,
                version: TlsVersion::Tls12,
                len: 4,
            },
            msg: vec![TlsMessage::Handshake(TlsMessageHandshake::ServerDone(
                empty,
            ))],
        };
        assert_eq!(parse_tls_plaintext(&bytes), Ok((empty, expected)));
    }

    // client response, composed of 3 records:
    // - Client Key Exchange
    // - Change Cipher Spec
    // - an encrypted handshake message
    #[rustfmt::skip]
static CLIENT_REPLY1: &'static [u8] = &[
    0x16, 0x03, 0x03, 0x00, 0x46, 0x10, 0x00, 0x00, 0x42, 0x41, 0x04, 0x22,
    0xd3, 0xf9, 0xbf, 0xbb, 0x7e, 0x34, 0xf9, 0x95, 0x68, 0x2e, 0xe2, 0xf8,
    0xf3, 0xf8, 0x08, 0x9c, 0x78, 0x32, 0x81, 0xa8, 0x28, 0x33, 0x5e, 0x46,
    0x11, 0xf2, 0x31, 0x2c, 0x9f, 0x77, 0xda, 0xc0, 0x88, 0xb5, 0xb4, 0x19,
    0xc0, 0x97, 0x3d, 0xe0, 0x99, 0x5c, 0xec, 0x1e, 0xbc, 0x32, 0x62, 0x8e,
    0x47, 0xc4, 0x7c, 0xcb, 0x31, 0x38, 0x5a, 0xed, 0x09, 0x1f, 0x82, 0xb1,
    0xb3, 0xce, 0x43, 0x14, 0x03, 0x03, 0x00, 0x01, 0x01, 0x16, 0x03, 0x03,
    0x00, 0x28, 0x00, 0x74, 0x47, 0x18, 0x4c, 0x5f, 0xbf, 0x65, 0xfe, 0xb9,
    0x34, 0xcf, 0x21, 0x8d, 0x6c, 0xd6, 0x99, 0xac, 0x24, 0xd3, 0x5a, 0x54,
    0x44, 0x05, 0x41, 0x7b, 0x1a, 0x25, 0xe6, 0xbf, 0xe0, 0x82, 0x95, 0x72,
    0x38, 0x7a, 0xa5, 0xd8, 0xf3, 0x72
];

    #[test]
    fn test_tls_record_clientkeyexchange() {
        let empty = &b""[..];
        let bytes = &CLIENT_REPLY1[0..75];
        let expected = TlsPlaintext {
            hdr: TlsRecordHeader {
                record_type: TlsRecordType::Handshake,
                version: TlsVersion::Tls12,
                len: 70,
            },
            msg: vec![TlsMessage::Handshake(
                TlsMessageHandshake::ClientKeyExchange(TlsClientKeyExchangeContents::Unknown(
                    &bytes[9..],
                )),
            )],
        };
        assert_eq!(parse_tls_plaintext(&bytes), Ok((empty, expected)));
    }

    #[test]
    fn test_tls_record_changecipherspec() {
        let empty = &b""[..];
        let bytes = &CLIENT_REPLY1[75..81];
        let expected = TlsPlaintext {
            hdr: TlsRecordHeader {
                record_type: TlsRecordType::ChangeCipherSpec,
                version: TlsVersion::Tls12,
                len: 1,
            },
            msg: vec![TlsMessage::ChangeCipherSpec],
        };
        assert_eq!(parse_tls_plaintext(&bytes), Ok((empty, expected)));
    }

    #[test]
    fn test_tls_record_encryptedhandshake() {
        let empty = &b""[..];
        let bytes = &CLIENT_REPLY1[81..];
        let expected = TlsEncrypted {
            hdr: TlsRecordHeader {
                record_type: TlsRecordType::Handshake,
                version: TlsVersion::Tls12,
                len: bytes.len() as u16 - 5,
            },
            msg: TlsEncryptedContent { blob: &bytes[5..] },
        };
        assert_eq!(parse_tls_encrypted(&bytes), Ok((empty, expected)));
    }

    #[rustfmt::skip]
static SERVER_HELLO1: &'static [u8] = &[
    0x16, 0x03, 0x03, 0x00, 0x3b, 0x02, 0x00, 0x00, 0x37, 0x03, 0x03, 0x57,
    0xc4, 0x57, 0xda, 0x9c, 0xd3, 0x24, 0x6d, 0x9d, 0x02, 0x26, 0xa2, 0xe5,
    0x9a, 0xe8, 0xa5, 0x6f, 0x40, 0xad, 0x94, 0x30, 0xba, 0x49, 0x05, 0x3a,
    0x1e, 0x1b, 0xe1, 0x94, 0xa1, 0xba, 0x41, 0x00, 0xc0, 0x2f, 0x00, 0x00,
    0x0f, 0xff, 0x01, 0x00, 0x01, 0x00, 0x00, 0x23, 0x00, 0x00, 0x00, 0x0b,
    0x00, 0x02, 0x01, 0x00
];

    #[test]
    fn test_tls_record_invalid_recordlength() {
        let mut v = SERVER_HELLO1.to_vec();
        let bytes = v.as_mut_slice();
        bytes[4] = 0xff; // make record incomplete (longer than data)
        let expected = Err(Err::Incomplete(Needed::Size(255)));
        let res = parse_tls_plaintext(&bytes);
        assert_eq!(res, expected);
    }

    #[test]
    fn test_tls_record_invalid_recordlength2() {
        let mut v = SERVER_HELLO1.to_vec();
        let bytes = v.as_mut_slice();
        bytes[4] = 0x00; // make record incomplete (shorter than data)
        let res = parse_tls_plaintext(&bytes);
        assert!(res.is_err());
    }

    #[test]
    fn test_tls_record_invalid_messagelength() {
        let mut v = SERVER_HELLO1.to_vec();
        let bytes = v.as_mut_slice();
        bytes[8] = 0xff; // create message larger than record
        let res = parse_tls_plaintext(&bytes);
        assert!(res.is_err());
    }

    #[test]
    fn test_tls_record_invalid_messagelength2() {
        let mut v = SERVER_HELLO1.to_vec();
        let bytes = v.as_mut_slice();
        bytes[8] = 0x01; // create message shorter than record
        let res = parse_tls_plaintext(&bytes);
        assert!(res.is_err());
    }

    #[rustfmt::skip]
static SERVER_CERTIFICATE_REQUEST_NOCA: &'static [u8] = &[
    0x16, 0x03, 0x03, 0x00, 0x2a, 0x0d, 0x00, 0x00, 0x26, 0x03, 0x01, 0x02,
    0x40, 0x00, 0x1e, 0x06, 0x01, 0x06, 0x02, 0x06, 0x03, 0x05, 0x01, 0x05,
    0x02, 0x05, 0x03, 0x04, 0x01, 0x04, 0x02, 0x04, 0x03, 0x03, 0x01, 0x03,
    0x02, 0x03, 0x03, 0x02, 0x01, 0x02, 0x02, 0x02, 0x03, 0x00, 0x00
];

    #[test]
    fn test_tls_record_cert_request_noca() {
        let empty = &b""[..];
        let bytes = SERVER_CERTIFICATE_REQUEST_NOCA;
        let expected = TlsPlaintext {
            hdr: TlsRecordHeader {
                record_type: TlsRecordType::Handshake,
                version: TlsVersion::Tls12,
                len: bytes.len() as u16 - 5,
            },
            msg: vec![TlsMessage::Handshake(
                TlsMessageHandshake::CertificateRequest(TlsCertificateRequestContents {
                    cert_types: vec![0x01, 0x02, 0x40],
                    sig_hash_algs: Some(vec![
                        0x0601, 0x0602, 0x0603, 0x0501, 0x0502, 0x0503, 0x0401, 0x0402, 0x0403,
                        0x0301, 0x0302, 0x0303, 0x0201, 0x0202, 0x0203,
                    ]),
                    unparsed_ca: vec![],
                }),
            )],
        };
        assert_eq!(parse_tls_plaintext(&bytes), Ok((empty, expected)));
    }

    #[rustfmt::skip]
static SERVER_CERTIFICATE_REQUEST_CA: &'static [u8] = &[
    0x16, 0x03, 0x03, 0x00, 0x73, 0x0d, 0x00, 0x00, 0x6f, 0x03, 0x01, 0x02,
    0x40, 0x00, 0x1e, 0x06, 0x01, 0x06, 0x02, 0x06, 0x03, 0x05, 0x01, 0x05,
    0x02, 0x05, 0x03, 0x04, 0x01, 0x04, 0x02, 0x04, 0x03, 0x03, 0x01, 0x03,
    0x02, 0x03, 0x03, 0x02, 0x01, 0x02, 0x02, 0x02, 0x03, 0x00, 0x49, 0x00,
    0x47, 0x30, 0x45, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06,
    0x13, 0x02, 0x46, 0x52, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04,
    0x08, 0x0c, 0x0a, 0x53, 0x6f, 0x6d, 0x65, 0x2d, 0x53, 0x74, 0x61, 0x74,
    0x65, 0x31, 0x21, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c, 0x18,
    0x49, 0x6e, 0x74, 0x65, 0x72, 0x6e, 0x65, 0x74, 0x20, 0x57, 0x69, 0x64,
    0x67, 0x69, 0x74, 0x73, 0x20, 0x50, 0x74, 0x79, 0x20, 0x4c, 0x74, 0x64
];

    #[test]
    fn test_tls_record_cert_request_ca() {
        let empty = &b""[..];
        let bytes = SERVER_CERTIFICATE_REQUEST_CA;
        let ca1 = &bytes[49..];
        let expected = TlsPlaintext {
            hdr: TlsRecordHeader {
                record_type: TlsRecordType::Handshake,
                version: TlsVersion::Tls12,
                len: bytes.len() as u16 - 5,
            },
            msg: vec![TlsMessage::Handshake(
                TlsMessageHandshake::CertificateRequest(TlsCertificateRequestContents {
                    cert_types: vec![0x01, 0x02, 0x40],
                    sig_hash_algs: Some(vec![
                        0x0601, 0x0602, 0x0603, 0x0501, 0x0502, 0x0503, 0x0401, 0x0402, 0x0403,
                        0x0301, 0x0302, 0x0303, 0x0201, 0x0202, 0x0203,
                    ]),
                    unparsed_ca: vec![ca1],
                }),
            )],
        };
        assert_eq!(parse_tls_plaintext(&bytes), Ok((empty, expected)));
    }

    #[rustfmt::skip]
static SERVER_STATUS_RESPONSE: &'static [u8] = &[
    0x16, 0x00, 0x06, 0x3f, 0x01, 0x00, 0x06, 0x3b, 0x30, 0x82, 0x06, 0x37,
    0x0a, 0x01, 0x00, 0xa0, 0x82, 0x06, 0x30, 0x30, 0x82, 0x06, 0x2c, 0x06,
    0x09, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x30, 0x01, 0x01, 0x04, 0x82,
    0x06, 0x1d, 0x30, 0x82, 0x06, 0x19, 0x30, 0x81, 0x9e, 0xa2, 0x16, 0x04,
    0x14, 0x9b, 0xe1, 0x51, 0x6a, 0xe3, 0xbe, 0x1a, 0x97, 0x21, 0x2b, 0x67,
    0xb1, 0x07, 0xd4, 0x54, 0x6c, 0x4f, 0xae, 0x53, 0x3c, 0x18, 0x0f, 0x32,
    0x30, 0x31, 0x36, 0x31, 0x30, 0x31, 0x32, 0x32, 0x33, 0x34, 0x32, 0x35,
    0x31, 0x5a, 0x30, 0x73, 0x30, 0x71, 0x30, 0x49, 0x30, 0x09, 0x06, 0x05,
    0x2b, 0x0e, 0x03, 0x02, 0x1a, 0x05, 0x00, 0x04, 0x14, 0x74, 0x24, 0x14,
    0x67, 0x06, 0x9f, 0xf5, 0xe0, 0x98, 0x3f, 0x5e, 0x3e, 0x1a, 0x6b, 0xa0,
    0x65, 0x2a, 0x54, 0x15, 0x75, 0x04, 0x14, 0x01, 0x59, 0xab, 0xe7, 0xdd,
    0x3a, 0x0b, 0x59, 0xa6, 0x64, 0x63, 0xd6, 0xcf, 0x20, 0x07, 0x57, 0xd5,
    0x91, 0xe7, 0x6a, 0x02, 0x10, 0x7e, 0x91, 0xc2, 0x1d, 0xe5, 0x0b, 0x21,
    0x74, 0x6d, 0x19, 0x78, 0xfa, 0x4b, 0xde, 0xbe, 0x48, 0x80, 0x00, 0x18,
    0x0f, 0x32, 0x30, 0x31, 0x36, 0x31, 0x30, 0x31, 0x32, 0x32, 0x33, 0x34,
    0x32, 0x35, 0x31, 0x5a, 0xa0, 0x11, 0x18, 0x0f, 0x32, 0x30, 0x31, 0x36,
    0x31, 0x30, 0x31, 0x39, 0x32, 0x33, 0x34, 0x32, 0x35, 0x31, 0x5a, 0x30,
    0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x05,
    0x05, 0x00, 0x03, 0x82, 0x01, 0x01, 0x00, 0xaa, 0x99, 0x41, 0xfe, 0x76,
    0x22, 0x24, 0x0c, 0x9e, 0xdb, 0x2a, 0xc7, 0xad, 0xe5, 0xf5, 0xf7, 0x0f,
    0x40, 0x2b, 0x04, 0x42, 0xb3, 0xe6, 0x0d, 0x4d, 0x52, 0xef, 0x0e, 0xd0,
    0xe7, 0x96, 0x99, 0x14, 0x95, 0xef, 0xda, 0x98, 0x25, 0x08, 0xda, 0x7f,
    0x8e, 0xbf, 0x0a, 0x63, 0x7c, 0xb0, 0x9f, 0x99, 0x41, 0x21, 0xd7, 0xf9,
    0x11, 0x0e, 0x60, 0x41, 0x24, 0x03, 0x87, 0x15, 0x08, 0xc7, 0x60, 0xe8,
    0x1c, 0xc6, 0x21, 0x7e, 0xc8, 0x08, 0x89, 0x7f, 0xbc, 0xa3, 0x75, 0x70,
    0x5c, 0xc6, 0x24, 0x24, 0x7a, 0x85, 0x49, 0x40, 0xdb, 0x20, 0x18, 0x75,
    0x0f, 0x94, 0x99, 0xb9, 0xcc, 0x4d, 0x8f, 0x06, 0xc8, 0xdf, 0x3a, 0xfc,
    0x33, 0x6d, 0x06, 0x9b, 0x00, 0xf3, 0x73, 0xfd, 0x04, 0x9a, 0xe0, 0x3c,
    0x2d, 0x14, 0xbd, 0x96, 0x8e, 0xf8, 0x01, 0xab, 0xb9, 0xc9, 0x0d, 0xb8,
    0xa6, 0x3f, 0x16, 0x07, 0x0e, 0x46, 0x7b, 0x76, 0xe5, 0x48, 0xfe, 0xc2,
    0xf6, 0xfa, 0x6c, 0xd9, 0x2e, 0xb4, 0x25, 0x20, 0x84, 0xd2, 0x6c, 0x8b,
    0xc6, 0x48, 0x14, 0x2d, 0x17, 0x06, 0x91, 0xd9, 0x66, 0x73, 0xf4, 0x47,
    0x2e, 0x81, 0x18, 0xa4, 0x72, 0xf1, 0x33, 0x12, 0x06, 0x3d, 0x12, 0xb6,
    0x59, 0x8f, 0xf8, 0xa2, 0x74, 0x74, 0x29, 0x69, 0xad, 0xc9, 0x1d, 0xc9,
    0x51, 0x08, 0x91, 0xf6, 0xda, 0xd6, 0xa6, 0x72, 0x46, 0xc4, 0x2a, 0x46,
    0x40, 0xf0, 0xa2, 0xb7, 0xd7, 0x3a, 0x8d, 0x47, 0x5b, 0xe0, 0xf3, 0x06,
    0x13, 0xf8, 0x40, 0xfa, 0xef, 0x70, 0x83, 0xfa, 0x9c, 0x5e, 0xfb, 0xe9,
    0x2d, 0xbd, 0x25, 0x37, 0x58, 0xff, 0xc1, 0xa1, 0x2a, 0x47, 0xbb, 0x81,
    0x31, 0x83, 0x74, 0xb1, 0x64, 0xd3, 0xf4, 0x1f, 0x88, 0x08, 0x9b, 0x4d,
    0x52, 0x5d, 0x7b, 0xcf, 0x82, 0x1f, 0xd7, 0x05, 0x79, 0x1d, 0x95, 0xa0,
    0x82, 0x04, 0x60, 0x30, 0x82, 0x04, 0x5c, 0x30, 0x82, 0x04, 0x58, 0x30,
    0x82, 0x03, 0x40, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x10, 0x30, 0x69,
    0x79, 0x69, 0x01, 0x5d, 0xf2, 0x7c, 0x9f, 0x97, 0x0c, 0xd1, 0x93, 0xb4,
    0x56, 0x38, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d,
    0x01, 0x01, 0x0b, 0x05, 0x00, 0x30, 0x77, 0x31, 0x0b, 0x30, 0x09, 0x06,
    0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x1d, 0x30, 0x1b,
    0x06, 0x03, 0x55, 0x04, 0x0a, 0x13, 0x14, 0x53, 0x79, 0x6d, 0x61, 0x6e,
    0x74, 0x65, 0x63, 0x20, 0x43, 0x6f, 0x72, 0x70, 0x6f, 0x72, 0x61, 0x74,
    0x69, 0x6f, 0x6e, 0x31, 0x1f, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x04, 0x0b,
    0x13, 0x16, 0x53, 0x79, 0x6d, 0x61, 0x6e, 0x74, 0x65, 0x63, 0x20, 0x54,
    0x72, 0x75, 0x73, 0x74, 0x20, 0x4e, 0x65, 0x74, 0x77, 0x6f, 0x72, 0x6b,
    0x31, 0x28, 0x30, 0x26, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x1f, 0x53,
    0x79, 0x6d, 0x61, 0x6e, 0x74, 0x65, 0x63, 0x20, 0x43, 0x6c, 0x61, 0x73,
    0x73, 0x20, 0x33, 0x20, 0x45, 0x56, 0x20, 0x53, 0x53, 0x4c, 0x20, 0x43,
    0x41, 0x20, 0x2d, 0x20, 0x47, 0x33, 0x30, 0x1e, 0x17, 0x0d, 0x31, 0x36,
    0x30, 0x38, 0x32, 0x34, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5a, 0x17,
    0x0d, 0x31, 0x36, 0x31, 0x31, 0x32, 0x32, 0x32, 0x33, 0x35, 0x39, 0x35,
    0x39, 0x5a, 0x30, 0x39, 0x31, 0x37, 0x30, 0x35, 0x06, 0x03, 0x55, 0x04,
    0x03, 0x13, 0x2e, 0x53, 0x79, 0x6d, 0x61, 0x6e, 0x74, 0x65, 0x63, 0x20,
    0x43, 0x6c, 0x61, 0x73, 0x73, 0x20, 0x33, 0x20, 0x45, 0x56, 0x20, 0x53,
    0x53, 0x4c, 0x20, 0x43, 0x41, 0x20, 0x2d, 0x20, 0x47, 0x33, 0x20, 0x4f,
    0x43, 0x53, 0x50, 0x20, 0x52, 0x65, 0x73, 0x70, 0x6f, 0x6e, 0x64, 0x65,
    0x72, 0x30, 0x82, 0x01, 0x22, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48,
    0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0f,
    0x00, 0x30, 0x82, 0x01, 0x0a, 0x02, 0x82, 0x01, 0x01, 0x00, 0xbe, 0x8e,
    0xc9, 0x7c, 0x69, 0xd3, 0x09, 0x45, 0xd4, 0x5f, 0xa1, 0xc7, 0xe7, 0x8d,
    0xdd, 0xbe, 0x9e, 0x17, 0x01, 0x8a, 0x6c, 0xb0, 0x57, 0xb9, 0x98, 0xfb,
    0x2f, 0x07, 0xa4, 0xde, 0xc9, 0x0f, 0x4b, 0x9f, 0x26, 0xc9, 0x77, 0x90,
    0xe6, 0xc0, 0xd0, 0xc2, 0xef, 0xc2, 0xc9, 0x3b, 0x4b, 0x86, 0xe0, 0xaf,
    0xc7, 0x87, 0xef, 0x9a, 0xbf, 0x26, 0x33, 0x95, 0x55, 0xf2, 0x7b, 0xa6,
    0x95, 0x86, 0x7d, 0x0d, 0x91, 0x82, 0xa8, 0x2c, 0x8a, 0xe5, 0x5c, 0x60,
    0xb7, 0x98, 0x01, 0xec, 0xd0, 0x76, 0x2d, 0xd7, 0x3e, 0xaf, 0x6b, 0xc0,
    0x35, 0xd0, 0x3a, 0xc1, 0xdf, 0x46, 0x84, 0xdf, 0x98, 0xdf, 0x09, 0xec,
    0xdf, 0x82, 0xc4, 0x0f, 0x05, 0xc9, 0xdb, 0x66, 0x87, 0xb6, 0x4d, 0x25,
    0x7d, 0x23, 0x43, 0x48, 0x71, 0x1c, 0x86, 0x21, 0xb2, 0x4f, 0xec, 0xf7,
    0x03, 0xd1, 0xf4, 0x4b, 0x74, 0x13, 0x03, 0xd7, 0x85, 0xfe, 0xcd, 0xf5,
    0xed, 0xf4, 0x89, 0xb1, 0x64, 0x1a, 0x2c, 0x4f, 0x2f, 0xee, 0xc5, 0x98,
    0xec, 0xb7, 0xee, 0xab, 0x10, 0x2e, 0x8b, 0x5b, 0x59, 0x7e, 0x97, 0xcc,
    0xc9, 0xd6, 0xa7, 0xa5, 0x3a, 0xde, 0x5a, 0x1a, 0xe6, 0x17, 0x17, 0xba,
    0xc8, 0x94, 0x1b, 0x2e, 0xbc, 0x28, 0xb2, 0x5a, 0x6b, 0x0a, 0x81, 0x72,
    0x51, 0xdd, 0x07, 0xed, 0x43, 0xcb, 0xde, 0xf4, 0xee, 0x41, 0x48, 0x56,
    0x03, 0x8b, 0x9b, 0x1d, 0xc4, 0xe2, 0x3b, 0xae, 0x9f, 0xe2, 0xc5, 0x63,
    0xca, 0xd9, 0x69, 0x30, 0x38, 0x87, 0x7e, 0x57, 0xaa, 0xa3, 0x4d, 0x30,
    0x5f, 0xe7, 0x4d, 0xe9, 0x5c, 0xc1, 0x32, 0xec, 0x58, 0xc7, 0x88, 0xea,
    0xa5, 0xd2, 0x92, 0xe2, 0x17, 0x8f, 0x9b, 0x25, 0xbf, 0xc5, 0xac, 0x16,
    0xdb, 0xde, 0xf2, 0x38, 0xf8, 0xf7, 0xe7, 0xe2, 0xe6, 0xcf, 0xdf, 0x9b,
    0x91, 0x7f, 0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x82, 0x01, 0x1c, 0x30,
    0x82, 0x01, 0x18, 0x30, 0x0f, 0x06, 0x09, 0x2b, 0x06, 0x01, 0x05, 0x05,
    0x07, 0x30, 0x01, 0x05, 0x04, 0x02, 0x05, 0x00, 0x30, 0x22, 0x06, 0x03,
    0x55, 0x1d, 0x11, 0x04, 0x1b, 0x30, 0x19, 0xa4, 0x17, 0x30, 0x15, 0x31,
    0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x0a, 0x54, 0x47,
    0x56, 0x2d, 0x44, 0x2d, 0x31, 0x37, 0x31, 0x39, 0x30, 0x1f, 0x06, 0x03,
    0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x01, 0x59, 0xab,
    0xe7, 0xdd, 0x3a, 0x0b, 0x59, 0xa6, 0x64, 0x63, 0xd6, 0xcf, 0x20, 0x07,
    0x57, 0xd5, 0x91, 0xe7, 0x6a, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e,
    0x04, 0x16, 0x04, 0x14, 0x9b, 0xe1, 0x51, 0x6a, 0xe3, 0xbe, 0x1a, 0x97,
    0x21, 0x2b, 0x67, 0xb1, 0x07, 0xd4, 0x54, 0x6c, 0x4f, 0xae, 0x53, 0x3c,
    0x30, 0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02,
    0x30, 0x00, 0x30, 0x6e, 0x06, 0x03, 0x55, 0x1d, 0x20, 0x04, 0x67, 0x30,
    0x65, 0x30, 0x63, 0x06, 0x0b, 0x60, 0x86, 0x48, 0x01, 0x86, 0xf8, 0x45,
    0x01, 0x07, 0x17, 0x03, 0x30, 0x54, 0x30, 0x26, 0x06, 0x08, 0x2b, 0x06,
    0x01, 0x05, 0x05, 0x07, 0x02, 0x01, 0x16, 0x1a, 0x68, 0x74, 0x74, 0x70,
    0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x73, 0x79, 0x6d, 0x61, 0x75,
    0x74, 0x68, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x63, 0x70, 0x73, 0x30, 0x2a,
    0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x02, 0x02, 0x30, 0x1e,
    0x1a, 0x1c, 0x20, 0x20, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77,
    0x77, 0x77, 0x2e, 0x73, 0x79, 0x6d, 0x61, 0x75, 0x74, 0x68, 0x2e, 0x63,
    0x6f, 0x6d, 0x2f, 0x72, 0x70, 0x61, 0x30, 0x13, 0x06, 0x03, 0x55, 0x1d,
    0x25, 0x04, 0x0c, 0x30, 0x0a, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05,
    0x07, 0x03, 0x09, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01,
    0xff, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x0d, 0x06, 0x09, 0x2a,
    0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x03, 0x82,
    0x01, 0x01, 0x00, 0x55, 0x8a, 0xbf, 0xb5, 0x01, 0xee, 0xc2, 0x12, 0x7f,
    0xa4, 0xf9, 0x8e, 0xec, 0xb6, 0x8c, 0x40, 0x10, 0x91, 0x7b, 0x84, 0xa3,
    0xc5, 0x3c, 0x12, 0x5d, 0xc4, 0x79, 0x55, 0xbf, 0xf9, 0xf0, 0x74, 0xfd,
    0x9c, 0xd6, 0xae, 0x9e, 0x4b, 0x90, 0x04, 0x2e, 0xfd, 0x8f, 0xf8, 0x6b,
    0x6c, 0x0b, 0x3f, 0x2b, 0x94, 0xaa, 0xc2, 0xe3, 0x07, 0xc6, 0xbe, 0xcc,
    0x33, 0xe1, 0x68, 0x7e, 0x43, 0x8b, 0xd9, 0x36, 0x17, 0xb0, 0x32, 0x87,
    0xec, 0xe1, 0x10, 0x4d, 0x72, 0x26, 0xad, 0xd5, 0xa5, 0xa2, 0x9a, 0xf3,
    0x54, 0x21, 0x78, 0x0e, 0xdb, 0xa6, 0xba, 0xa5, 0x2e, 0x7f, 0x4d, 0xc6,
    0x5a, 0xcd, 0xa5, 0xfd, 0x38, 0x0d, 0x34, 0x9d, 0xe0, 0x88, 0xd7, 0x51,
    0x40, 0xd8, 0x73, 0xf3, 0x0c, 0xec, 0x53, 0x8f, 0x4e, 0x74, 0xda, 0xd3,
    0x3c, 0x67, 0x23, 0xba, 0x0f, 0x1c, 0xa8, 0x46, 0x32, 0x19, 0x3a, 0xbc,
    0xad, 0x4d, 0x3e, 0xd3, 0x4f, 0x89, 0xa9, 0xe9, 0xb8, 0x00, 0x1e, 0x1d,
    0x04, 0xb5, 0x7f, 0x31, 0xfd, 0xc0, 0x9e, 0xd0, 0xa8, 0xad, 0x46, 0xab,
    0x06, 0x91, 0x3f, 0xf4, 0xae, 0xce, 0xb7, 0x47, 0xb2, 0x4d, 0xcf, 0x93,
    0x5b, 0x4c, 0x62, 0xf2, 0x43, 0xd6, 0xd1, 0xde, 0x27, 0x33, 0xe0, 0x64,
    0xa9, 0x0e, 0x83, 0xa0, 0x93, 0x3c, 0xe8, 0xb2, 0xa0, 0x60, 0xb5, 0xcd,
    0x48, 0x51, 0x1e, 0x66, 0x00, 0x46, 0xcd, 0xe5, 0x22, 0x9f, 0x55, 0x2a,
    0xdc, 0xea, 0x61, 0xb3, 0xe5, 0x84, 0x81, 0x36, 0x29, 0x06, 0x6a, 0x1f,
    0xe0, 0x48, 0x11, 0x53, 0xb0, 0x60, 0xcf, 0x88, 0x06, 0xc7, 0xbc, 0xb9,
    0x57, 0xbe, 0x06, 0x12, 0xbf, 0x33, 0x8d, 0xc9, 0xe6, 0xa7, 0x36, 0x44,
    0xb8, 0x67, 0x0b, 0xa7, 0xe3, 0x0e, 0xbb, 0xdf, 0x5c, 0xd8, 0x56, 0xcd,
    0x25, 0x44, 0x93, 0xc9, 0x56, 0x22, 0xea
];

    #[test]
    fn test_tls_message_status_response() {
        let empty = &b""[..];
        let bytes = SERVER_STATUS_RESPONSE;
        let blob = &bytes[8..];
        let expected = vec![TlsMessage::Handshake(
            TlsMessageHandshake::CertificateStatus(TlsCertificateStatusContents {
                status_type: 1,
                blob: blob,
            }),
        )];
        let hdr = TlsRecordHeader {
            record_type: TlsRecordType::Handshake,
            version: TlsVersion(0),
            len: 0x0,
        };
        let res = parse_tls_record_with_header(&bytes, &hdr);
        assert_eq!(res, Ok((empty, expected)));
    }
} // mod tls_handshake
