/*
 * Copyright (C) 2009-2013 The Project Lombok Authors.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package lombok.delombok;

import java.io.IOException;
import java.io.Writer;
import java.util.Date;
import java.util.List;

import javax.tools.JavaFileObject;

import lombok.javac.CommentInfo;

import com.sun.tools.javac.tree.JCTree.JCCompilationUnit;

public class DelombokResult {
	private final List<CommentInfo> comments;
	private final JCCompilationUnit compilationUnit;
	private final boolean changed;
	private final FormatPreferences formatPreferences;
	
	public DelombokResult(List<CommentInfo> comments, JCCompilationUnit compilationUnit, boolean changed, FormatPreferences formatPreferences) {
		this.comments = comments;
		this.compilationUnit = compilationUnit;
		this.changed = changed;
		this.formatPreferences = formatPreferences;
	}
	
	public void print(Writer out) throws IOException {
		if (!changed) {
			CharSequence content = getContent();
			if (content != null) {
				out.write(content.toString());
				return;
			}
		}
		
		if (formatPreferences.generateDelombokComment()) {
			out.write("// Generated by delombok at ");
			out.write(String.valueOf(new Date()));
			out.write(System.getProperty("line.separator"));
		}
		
		com.sun.tools.javac.util.List<CommentInfo> comments_;
		if (comments instanceof com.sun.tools.javac.util.List) comments_ = (com.sun.tools.javac.util.List<CommentInfo>) comments;
		else comments_ = com.sun.tools.javac.util.List.from(comments.toArray(new CommentInfo[0]));
		
		FormatPreferences preferences = new FormatPreferenceScanner().scan(formatPreferences, getContent());
		compilationUnit.accept(new PrettyCommentsPrinter(out, compilationUnit, comments_, preferences));
	}

	private CharSequence getContent() throws IOException {
		JavaFileObject sourceFile = compilationUnit.getSourceFile();
		if (sourceFile == null) return null;
		return sourceFile.getCharContent(true);
	}
	
	public boolean isChanged() {
		return changed;
	}
}