/**
 * Author......: See docs/credits.txt
 * License.....: MIT
 */

#include "common.h"
#include "types.h"
#include "modules.h"
#include "bitops.h"
#include "convert.h"
#include "shared.h"
#include "memory.h"

static const u32   ATTACK_EXEC    = ATTACK_EXEC_OUTSIDE_KERNEL;
static const u32   DGST_POS0      = 0;
static const u32   DGST_POS1      = 1;
static const u32   DGST_POS2      = 2;
static const u32   DGST_POS3      = 3;
static const u32   DGST_SIZE      = DGST_SIZE_4_16;
static const u32   HASH_CATEGORY  = HASH_CATEGORY_FDE;
static const char *HASH_NAME      = "LUKS v1 SHA-256 + Twofish";
static const u64   KERN_TYPE      = 14623; // old kernel used here
static const u32   OPTI_TYPE      = OPTI_TYPE_ZERO_BYTE
                                  | OPTI_TYPE_SLOW_HASH_SIMD_LOOP;
static const u64   OPTS_TYPE      = OPTS_TYPE_STOCK_MODULE
                                  | OPTS_TYPE_PT_GENERATE_LE;
static const u32   SALT_TYPE      = SALT_TYPE_EMBEDDED;
static const char *ST_PASS        = "hashcat";
static const char *ST_HASH        = "$luks$1$sha256$twofish$cbc-essiv:sha256$256$92561$3a9c59dc9aea663637ef9355a680e97c804dc77a890fe893c4393dd7341c2671$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$130f89976f8ab366002d7c0a6a2de1cc89f476fcec2acdb83e4df17508ff750e6dab348eda053e18cd23902e9a530edbe1b34f68ec0126b10320641abfcdd950c7e0c720d8da73949f5f91a0b3fb983e2f6fb9307dcd4b6aff12b766b561a1307f9c44005b9272bbff75209073c27cd3fd550873df31c8518ef9aa02e872de81c0fd31f32a2241b64ed74c6b907ec9350f828572650434d887b40caf50f3f752dfaee6698659c8cc32d141a1356c5784ca23237cb3e28200d3b004f7085262d2c86e381c7f18418b3274eb7513ecaf5e11ecdab21108e85c2d7a5a9dedc76686c099d5a8fd260765d3fed2cf7cf98bc4553564b83d4f32ff6c8bdadc32d4abd76614c86bcf22608413db5936861adcf2720ee3d60c51787d90e9d41cc0c7fc6132713cc5cea0ea79ed196748965eec9a4678ee8a93842c7d5712dc3ed4577ba685a9143e8e76c7d14feb3179364ad05d09e5846bf7dd7d63ce9b6bc5b942145ce9a44e16ce65f5290600c4e6b63c66660fefe23d8d27f574a7b63897ecb81bcd62ec7d8abd9f8014802b5c53c6cb28ddcc6a01cd0108ef05bd2409fe637c4ce9a0f3f4a76589b4a9ae6f160fae740e12fa30dd05f3c7aa57daf0226fda6074511005d4d3cd4d222c3f762dabb29cc1d150507c5b42f4c3eb6bef8b01afecdb70a9fd68f2dcac9576b6504ac0ba4adcdfd708524affe1e6e84efb69ed717a676d";

u32         module_attack_exec    (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return ATTACK_EXEC;     }
u32         module_dgst_pos0      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS0;       }
u32         module_dgst_pos1      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS1;       }
u32         module_dgst_pos2      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS2;       }
u32         module_dgst_pos3      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS3;       }
u32         module_dgst_size      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_SIZE;       }
u32         module_hash_category  (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return HASH_CATEGORY;   }
const char *module_hash_name      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return HASH_NAME;       }
u64         module_kern_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return KERN_TYPE;       }
u32         module_opti_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return OPTI_TYPE;       }
u64         module_opts_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return OPTS_TYPE;       }
u32         module_salt_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return SALT_TYPE;       }
const char *module_st_hash        (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return ST_HASH;         }
const char *module_st_pass        (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return ST_PASS;         }

#define LUKS_STRIPES        (                                   4000)
#define LUKS_SALT_LEN       (                                     32)
#define LUKS_SALT_HEX_LEN   (                      LUKS_SALT_LEN * 2)
#define LUKS_CT_LEN         (                                    512)
#define LUKS_CT_HEX_LEN     (                        LUKS_CT_LEN * 2)
#define LUKS_AF_MIN_LEN     (HC_LUKS_KEY_SIZE_128 / 8 * LUKS_STRIPES)
#define LUKS_AF_MIN_HEX_LEN (                    LUKS_AF_MIN_LEN * 2)
#define LUKS_AF_MAX_LEN     (HC_LUKS_KEY_SIZE_512 / 8 * LUKS_STRIPES)
#define LUKS_AF_MAX_HEX_LEN (                    LUKS_AF_MAX_LEN * 2)

typedef enum hc_luks_hash_type
{
  HC_LUKS_HASH_TYPE_SHA1      = 1,
  HC_LUKS_HASH_TYPE_SHA256    = 2,
  HC_LUKS_HASH_TYPE_SHA512    = 3,
  HC_LUKS_HASH_TYPE_RIPEMD160 = 4,
  HC_LUKS_HASH_TYPE_WHIRLPOOL = 5,

} hc_luks_hash_type_t;

typedef enum hc_luks_key_size
{
  HC_LUKS_KEY_SIZE_128 = 128,
  HC_LUKS_KEY_SIZE_256 = 256,
  HC_LUKS_KEY_SIZE_512 = 512,

} hc_luks_key_size_t;

typedef enum hc_luks_cipher_type
{
  HC_LUKS_CIPHER_TYPE_AES     = 1,
  HC_LUKS_CIPHER_TYPE_SERPENT = 2,
  HC_LUKS_CIPHER_TYPE_TWOFISH = 3,

} hc_luks_cipher_type_t;

typedef enum hc_luks_cipher_mode
{
  HC_LUKS_CIPHER_MODE_CBC_ESSIV_SHA256 = 1,
  HC_LUKS_CIPHER_MODE_CBC_PLAIN        = 2,
  HC_LUKS_CIPHER_MODE_CBC_PLAIN64      = 3,
  HC_LUKS_CIPHER_MODE_XTS_PLAIN        = 4,
  HC_LUKS_CIPHER_MODE_XTS_PLAIN64      = 5,

} hc_luks_cipher_mode_t;

typedef struct luks
{
  int hash_type;   // hc_luks_hash_type_t
  int key_size;    // hc_luks_key_size_t
  int cipher_type; // hc_luks_cipher_type_t
  int cipher_mode; // hc_luks_cipher_mode_t

  u32 ct_buf[LUKS_CT_LEN / 4];

  u32 af_buf[LUKS_AF_MAX_LEN / 4];
  u32 af_len;

} luks_t;

typedef struct luks_tmp
{
  u32 ipad32[8];
  u64 ipad64[8];

  u32 opad32[8];
  u64 opad64[8];

  u32 dgst32[32];
  u64 dgst64[16];

  u32 out32[32];
  u64 out64[16];

} luks_tmp_t;

static const char *SIGNATURE_LUKS = "$luks$1$sha256$twofish$";

u64 module_esalt_size (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra)
{
  const u64 esalt_size = (const u64) sizeof (luks_t);

  return esalt_size;
}

u64 module_tmp_size (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra)
{
  const u64 tmp_size = (const u64) sizeof (luks_tmp_t);

  return tmp_size;
}

u32 module_pw_max (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra)
{
  // this overrides the reductions of PW_MAX in case optimized kernel is selected
  // IOW, even in optimized kernel mode it support length 256

  const u32 pw_max = PW_MAX;

  return pw_max;
}

int module_hash_decode (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED void *digest_buf, MAYBE_UNUSED salt_t *salt, MAYBE_UNUSED void *esalt_buf, MAYBE_UNUSED void *hook_salt_buf, MAYBE_UNUSED hashinfo_t *hash_info, const char *line_buf, MAYBE_UNUSED const int line_len)
{
  luks_t *luks = (luks_t *) esalt_buf;

  hc_token_t token;

  token.token_cnt  = 7;

  token.signatures_cnt    = 1;
  token.signatures_buf[0] = SIGNATURE_LUKS;

  // signature with hash and cipher type
  token.len[0]     = 23;
  token.attr[0]    = TOKEN_ATTR_FIXED_LENGTH
                   | TOKEN_ATTR_VERIFY_SIGNATURE;

  // cipher mode
  token.sep[1]     = '$';
  token.len_min[1] = 9;
  token.len_max[1] = 16;
  token.attr[1]    = TOKEN_ATTR_VERIFY_LENGTH;

  // key size
  token.sep[2]     = '$';
  token.len_min[2] = 3;
  token.len_max[2] = 3;
  token.attr[2]    = TOKEN_ATTR_VERIFY_LENGTH
                   | TOKEN_ATTR_VERIFY_DIGIT;

  // iter
  token.sep[3]     = '$';
  token.len_min[3] = 1;
  token.len_max[3] = 10;
  token.attr[3]    = TOKEN_ATTR_VERIFY_LENGTH
                   | TOKEN_ATTR_VERIFY_DIGIT;

  // salt
  token.sep[4]     = '$';
  token.len_min[4] = LUKS_SALT_HEX_LEN;
  token.len_max[4] = LUKS_SALT_HEX_LEN;
  token.attr[4]    = TOKEN_ATTR_VERIFY_LENGTH
                   | TOKEN_ATTR_VERIFY_HEX;

  // af
  token.sep[5]     = '$';
  token.len_min[5] = LUKS_AF_MIN_HEX_LEN;
  token.len_max[5] = LUKS_AF_MAX_HEX_LEN;
  token.attr[5]    = TOKEN_ATTR_VERIFY_LENGTH
                   | TOKEN_ATTR_VERIFY_HEX;

  // ct
  token.sep[6]     = '$';
  token.len_min[6] = LUKS_CT_HEX_LEN;
  token.len_max[6] = LUKS_CT_HEX_LEN;
  token.attr[6]    = TOKEN_ATTR_VERIFY_LENGTH
                   | TOKEN_ATTR_VERIFY_HEX;

  const int rc_tokenizer = input_tokenizer ((const u8 *) line_buf, line_len, &token);

  if (rc_tokenizer != PARSER_OK) return (rc_tokenizer);

  // hash type

  luks->hash_type = HC_LUKS_HASH_TYPE_SHA256;

  // cipher type

  luks->cipher_type = HC_LUKS_CIPHER_TYPE_TWOFISH;

  // cipher mode

  const u8 *cipher_mode_pos = token.buf[1];

  if (strncmp ((const char *) cipher_mode_pos, "cbc-essiv:sha256", 16) == 0)
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_CBC_ESSIV_SHA256;
  }
  else if (strncmp ((const char *) cipher_mode_pos, "cbc-plain",    9) == 0)
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_CBC_PLAIN;
  }
  else if (strncmp ((const char *) cipher_mode_pos, "cbc-plain64", 11) == 0)
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_CBC_PLAIN64;
  }
  else if (strncmp ((const char *) cipher_mode_pos, "xts-plain",    9) == 0)
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_XTS_PLAIN;
  }
  else if (strncmp ((const char *) cipher_mode_pos, "xts-plain64", 11) == 0)
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_XTS_PLAIN64;
  }
  else
  {
    return (PARSER_LUKS_CIPHER_MODE);
  }

  // key size

  const u8 *key_size_pos = token.buf[2];

  const u32 key_size = hc_strtoul ((const char *) key_size_pos, NULL, 10);

  if (key_size == 128)
  {
    luks->key_size = HC_LUKS_KEY_SIZE_128;
  }
  else if (key_size == 256)
  {
    luks->key_size = HC_LUKS_KEY_SIZE_256;
  }
  else if (key_size == 512)
  {
    luks->key_size = HC_LUKS_KEY_SIZE_512;
  }
  else
  {
    return (PARSER_LUKS_KEY_SIZE);
  }

  // salt

  const u8 *salt_pos = token.buf[4];

  salt->salt_len = hex_decode (salt_pos, LUKS_SALT_HEX_LEN, (u8 *) salt->salt_buf);

  // iter

  const u8 *iter_pos = token.buf[3];

  const u32 iter = hc_strtoul ((const char *) iter_pos, NULL, 10);

  salt->salt_iter = iter - 1;

  // af

  u32 af_len = token.len[5] / 2;

  if (af_len != (key_size / 8) * LUKS_STRIPES) return (PARSER_LUKS_FILE_SIZE);

  const u8 *af_pos = token.buf[5];

  luks->af_len = hex_decode (af_pos, token.len[5], (u8 *) luks->af_buf);

  // ct

  u32 ct_len = token.len[6] / 2;

  if (ct_len != LUKS_CT_LEN) return (PARSER_LUKS_FILE_SIZE);

  const u8 *ct_pos = token.buf[6];

  hex_decode (ct_pos, LUKS_CT_HEX_LEN, (u8 *) luks->ct_buf);

  return (PARSER_OK);
}

int module_hash_encode (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const void *digest_buf, MAYBE_UNUSED const salt_t *salt, MAYBE_UNUSED const void *esalt_buf, MAYBE_UNUSED const void *hook_salt_buf, MAYBE_UNUSED const hashinfo_t *hash_info, char *line_buf, MAYBE_UNUSED const int line_size)
{
  luks_t *luks = (luks_t *) esalt_buf;

  // cipher mode

  char *cipher_mode = "";

  switch (luks->cipher_mode)
  {
    case HC_LUKS_CIPHER_MODE_CBC_ESSIV_SHA256: cipher_mode = "cbc-essiv:sha256"; break;
    case HC_LUKS_CIPHER_MODE_CBC_PLAIN:        cipher_mode = "cbc-plain";        break;
    case HC_LUKS_CIPHER_MODE_CBC_PLAIN64:      cipher_mode = "cbc-plain64";      break;
    case HC_LUKS_CIPHER_MODE_XTS_PLAIN:        cipher_mode = "xts-plain";        break;
    case HC_LUKS_CIPHER_MODE_XTS_PLAIN64:      cipher_mode = "xts-plain64";      break;
    default:                                                                     return 0;
  }

  // key size

  u32 key_size = 0;

  switch (luks->key_size)
  {
    case HC_LUKS_KEY_SIZE_128: key_size = 128; break;
    case HC_LUKS_KEY_SIZE_256: key_size = 256; break;
    case HC_LUKS_KEY_SIZE_512: key_size = 512; break;
    default:                                   return 0;
  }

  // iterations

  u32 iterations = salt->salt_iter + 1;

  // salt

  char salt_buf[LUKS_SALT_HEX_LEN + 1] = { 0 };

  hex_encode ((const u8 *) salt->salt_buf, LUKS_SALT_LEN, (u8 *) salt_buf);

  // af

  char af_buf[LUKS_AF_MAX_HEX_LEN + 1] = { 0 };

  hex_encode ((const u8 *) luks->af_buf, luks->af_len, (u8 *) af_buf);

  // ct

  char ct_buf[LUKS_CT_HEX_LEN + 1] = { 0 };

  hex_encode ((const u8 *) luks->ct_buf, LUKS_CT_LEN, (u8 *) ct_buf);

  // output

  int line_len = snprintf (line_buf, line_size, "%s%s$%u$%u$%s$%s$%s",
    SIGNATURE_LUKS,
    cipher_mode,
    key_size,
    iterations,
    salt_buf,
    af_buf,
    ct_buf);

  return line_len;
}

void module_init (module_ctx_t *module_ctx)
{
  module_ctx->module_context_size             = MODULE_CONTEXT_SIZE_CURRENT;
  module_ctx->module_interface_version        = MODULE_INTERFACE_VERSION_CURRENT;

  module_ctx->module_attack_exec              = module_attack_exec;
  module_ctx->module_benchmark_esalt          = MODULE_DEFAULT;
  module_ctx->module_benchmark_hook_salt      = MODULE_DEFAULT;
  module_ctx->module_benchmark_mask           = MODULE_DEFAULT;
  module_ctx->module_benchmark_charset        = MODULE_DEFAULT;
  module_ctx->module_benchmark_salt           = MODULE_DEFAULT;
  module_ctx->module_build_plain_postprocess  = MODULE_DEFAULT;
  module_ctx->module_deep_comp_kernel         = MODULE_DEFAULT;
  module_ctx->module_deprecated_notice        = MODULE_DEFAULT;
  module_ctx->module_dgst_pos0                = module_dgst_pos0;
  module_ctx->module_dgst_pos1                = module_dgst_pos1;
  module_ctx->module_dgst_pos2                = module_dgst_pos2;
  module_ctx->module_dgst_pos3                = module_dgst_pos3;
  module_ctx->module_dgst_size                = module_dgst_size;
  module_ctx->module_dictstat_disable         = MODULE_DEFAULT;
  module_ctx->module_esalt_size               = module_esalt_size;
  module_ctx->module_extra_buffer_size        = MODULE_DEFAULT;
  module_ctx->module_extra_tmp_size           = MODULE_DEFAULT;
  module_ctx->module_extra_tuningdb_block     = MODULE_DEFAULT;
  module_ctx->module_forced_outfile_format    = MODULE_DEFAULT;
  module_ctx->module_hash_binary_count        = MODULE_DEFAULT;
  module_ctx->module_hash_binary_parse        = MODULE_DEFAULT;
  module_ctx->module_hash_binary_save         = MODULE_DEFAULT;
  module_ctx->module_hash_decode_postprocess  = MODULE_DEFAULT;
  module_ctx->module_hash_decode_potfile      = MODULE_DEFAULT;
  module_ctx->module_hash_decode_zero_hash    = MODULE_DEFAULT;
  module_ctx->module_hash_decode              = module_hash_decode;
  module_ctx->module_hash_encode_status       = MODULE_DEFAULT;
  module_ctx->module_hash_encode_potfile      = MODULE_DEFAULT;
  module_ctx->module_hash_encode              = module_hash_encode;
  module_ctx->module_hash_init_selftest       = MODULE_DEFAULT;
  module_ctx->module_hash_mode                = MODULE_DEFAULT;
  module_ctx->module_hash_category            = module_hash_category;
  module_ctx->module_hash_name                = module_hash_name;
  module_ctx->module_hashes_count_min         = MODULE_DEFAULT;
  module_ctx->module_hashes_count_max         = MODULE_DEFAULT;
  module_ctx->module_hlfmt_disable            = MODULE_DEFAULT;
  module_ctx->module_hook_extra_param_size    = MODULE_DEFAULT;
  module_ctx->module_hook_extra_param_init    = MODULE_DEFAULT;
  module_ctx->module_hook_extra_param_term    = MODULE_DEFAULT;
  module_ctx->module_hook12                   = MODULE_DEFAULT;
  module_ctx->module_hook23                   = MODULE_DEFAULT;
  module_ctx->module_hook_salt_size           = MODULE_DEFAULT;
  module_ctx->module_hook_size                = MODULE_DEFAULT;
  module_ctx->module_jit_build_options        = MODULE_DEFAULT;
  module_ctx->module_jit_cache_disable        = MODULE_DEFAULT;
  module_ctx->module_kernel_accel_max         = MODULE_DEFAULT;
  module_ctx->module_kernel_accel_min         = MODULE_DEFAULT;
  module_ctx->module_kernel_loops_max         = MODULE_DEFAULT;
  module_ctx->module_kernel_loops_min         = MODULE_DEFAULT;
  module_ctx->module_kernel_threads_max       = MODULE_DEFAULT;
  module_ctx->module_kernel_threads_min       = MODULE_DEFAULT;
  module_ctx->module_kern_type                = module_kern_type;
  module_ctx->module_kern_type_dynamic        = MODULE_DEFAULT;
  module_ctx->module_opti_type                = module_opti_type;
  module_ctx->module_opts_type                = module_opts_type;
  module_ctx->module_outfile_check_disable    = MODULE_DEFAULT;
  module_ctx->module_outfile_check_nocomp     = MODULE_DEFAULT;
  module_ctx->module_potfile_custom_check     = MODULE_DEFAULT;
  module_ctx->module_potfile_disable          = MODULE_DEFAULT;
  module_ctx->module_potfile_keep_all_hashes  = MODULE_DEFAULT;
  module_ctx->module_pwdump_column            = MODULE_DEFAULT;
  module_ctx->module_pw_max                   = module_pw_max;
  module_ctx->module_pw_min                   = MODULE_DEFAULT;
  module_ctx->module_salt_max                 = MODULE_DEFAULT;
  module_ctx->module_salt_min                 = MODULE_DEFAULT;
  module_ctx->module_salt_type                = module_salt_type;
  module_ctx->module_separator                = MODULE_DEFAULT;
  module_ctx->module_st_hash                  = module_st_hash;
  module_ctx->module_st_pass                  = module_st_pass;
  module_ctx->module_tmp_size                 = module_tmp_size;
  module_ctx->module_unstable_warning         = MODULE_DEFAULT;
  module_ctx->module_warmup_disable           = MODULE_DEFAULT;
}
