/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file ab_ebook.c
 * \brief Evolution address book plugin
 */

#include <libedataserver/eds-version.h>

#include <libebook/e-book.h>
#include <libebook/e-contact.h>
#include <libebook/e-book-query.h>
#include <ffgtk.h>
#include <addressbook.h>

/**
 * \brief Get selected evolution address book
 * \return evolution address book
 */
static const gchar *ebookGetSelectedBook( void ) {
	return prefsGetString( getActiveProfile(), "/plugins/evolution/book" );
}

/**
 * \brief Set selected evolution address book
 * \param pnUri evolution address book
 */
static void ebookSetSelectedBook( gchar *pnUri ) {
	prefsSetString( getActiveProfile(), "/plugins/evolution/book", pnUri );
}

#if HAVE_EBOOK_V3
#include <libebook/e-book-client.h>
#include <libebook/e-book-query.h>

void updateAddressBook( void );

void readCallback( GObject *psSource, GAsyncResult *pnRes, gpointer pUserData ) {
	EBookClient *psClient = E_BOOK_CLIENT( psSource );
	EBookQuery *psQuery;
	const gchar *pnSexp = NULL;
	EContact *psContact;
	EContactPhoto *psPhoto;
	GdkPixbufLoader *psLoader;
	GSList *psContacts;
	GSList *psList;
	EContactAddress *psPrivateAddress = NULL;
	EContactAddress *psBusinessAddress = NULL;

	psQuery = e_book_query_any_field_contains( "" );
	if ( psQuery == NULL ) {
		Debug( KERN_WARNING, "Couldn't create query.\n" );
		return;
	}
	pnSexp = e_book_query_to_string( psQuery );

	if ( !e_book_client_get_contacts_sync( psClient, pnSexp, &psContacts, NULL, NULL ) ) {
		Debug( KERN_WARNING, "Couldn't get query results.\n" );
		e_book_query_unref( psQuery );
		g_object_unref( psClient );
		return;
	} else {
		Debug( KERN_DEBUG, "Got contacts\n" );
	}

	e_book_query_unref( psQuery );
	g_object_unref( psClient );

	if ( psContacts == NULL ) {
		Debug( KERN_WARNING, "No contacts found.\n" );
		return;
	}

	for ( psList = psContacts; psList != NULL; psList = psList -> next ) {
		GHashTable *psTable = NULL;
		const gchar *pnDisplayName;

		g_return_if_fail( E_IS_CONTACT( psList -> data ) );
		psContact = E_CONTACT( psList -> data );

		pnDisplayName = e_contact_get_const( psContact, E_CONTACT_FULL_NAME );

		if ( pnDisplayName == NULL ) {
			continue;
		}

		psTable = g_hash_table_new( NULL, NULL );

		AddInfo( psTable, PERSON_ID, e_contact_get_const( psContact, E_CONTACT_UID ) );
		AddInfo( psTable, PERSON_FIRST_NAME, e_contact_get_const( psContact, E_CONTACT_GIVEN_NAME ) );
		AddInfo( psTable, PERSON_LAST_NAME, e_contact_get_const( psContact, E_CONTACT_FAMILY_NAME ) );
		AddInfo( psTable, PERSON_DISPLAY_NAME, pnDisplayName );
		AddInfo( psTable, PERSON_COMPANY, e_contact_get_const( psContact, E_CONTACT_ORG ) );
		AddInfo( psTable, PERSON_TITLE, e_contact_get_const( psContact, E_CONTACT_TITLE ) );
		AddInfo( psTable, PERSON_BUSINESS_PHONE, e_contact_get_const( psContact, E_CONTACT_PHONE_BUSINESS ) );
		AddInfo( psTable, PERSON_BUSINESS_FAX, e_contact_get_const( psContact, E_CONTACT_PHONE_BUSINESS_FAX ) );
		AddInfo( psTable, PERSON_PRIVATE_PHONE, e_contact_get_const( psContact, E_CONTACT_PHONE_HOME ) );
		AddInfo( psTable, PERSON_PRIVATE_FAX, e_contact_get_const( psContact, E_CONTACT_PHONE_HOME_FAX ) );
		AddInfo( psTable, PERSON_PRIVATE_MOBILE, e_contact_get_const( psContact, E_CONTACT_PHONE_MOBILE ) );
		psPrivateAddress = e_contact_get( psContact, E_CONTACT_ADDRESS_HOME );
		if ( psPrivateAddress != NULL ) {
			AddInfo( psTable, PERSON_PRIVATE_STREET, psPrivateAddress -> street );
			AddInfo( psTable, PERSON_PRIVATE_CITY, psPrivateAddress -> locality );
			AddInfo( psTable, PERSON_PRIVATE_ZIPCODE, psPrivateAddress -> code );
			AddInfo( psTable, PERSON_PRIVATE_COUNTRY, psPrivateAddress -> country );
		}
		psBusinessAddress = e_contact_get( psContact, E_CONTACT_ADDRESS_WORK );
		if ( psBusinessAddress != NULL ) {
			AddInfo( psTable, PERSON_BUSINESS_STREET, psBusinessAddress -> street );
			AddInfo( psTable, PERSON_BUSINESS_CITY, psBusinessAddress -> locality );
			AddInfo( psTable, PERSON_BUSINESS_ZIPCODE, psBusinessAddress -> code );
			AddInfo( psTable, PERSON_BUSINESS_COUNTRY, psBusinessAddress -> country );
		}

		psPhoto = e_contact_get( psContact, E_CONTACT_PHOTO );
		if ( psPhoto != NULL ) {
			psLoader = gdk_pixbuf_loader_new();
			if ( gdk_pixbuf_loader_write( psLoader, psPhoto -> data.inlined.data, psPhoto -> data.inlined.length, NULL ) ) {
				AddInfo( psTable, PERSON_IMAGE, ( const gchar * ) gdk_pixbuf_loader_get_pixbuf( psLoader ) );
			}
			gdk_pixbuf_loader_close( psLoader, NULL );
			e_contact_photo_free( psPhoto );
		}

		AddPerson( psTable, FALSE );
		if ( psPrivateAddress != NULL ) {
			e_contact_address_free( psPrivateAddress );
		}
		if ( psBusinessAddress != NULL ) {
			e_contact_address_free( psBusinessAddress );
		}
		g_hash_table_destroy( psTable );
	}

	g_slist_free( psContacts );

	updateAddressBook();
}

/**
 * \brief Read ebook list
 * \return error code
 */
static int ebookReadBook( void ) {
	EBookClient *psClient;
	GError *psError = NULL;
	const gchar *pnUri = NULL;

	pnUri = ebookGetSelectedBook();
	if ( pnUri == NULL || strlen( pnUri ) == 0 ) {
		psClient = e_book_client_new_system( &psError );
	} else {
		psClient = e_book_client_new_from_uri( pnUri, &psError );
	}

	e_client_open( E_CLIENT( psClient ), TRUE, NULL, readCallback, NULL );

	return 0;
}

static void ebookSetImage( EContact *psContact, struct sPerson *psPerson ) {
	if ( psPerson -> pnNewImage != NULL ) {
		EContactPhoto sPhoto;
		guchar **ppnData;
		gsize *pnLength;

		sPhoto.type = E_CONTACT_PHOTO_TYPE_INLINED;
		ppnData = &sPhoto.data.inlined.data;
		pnLength = ( gsize * ) &sPhoto.data.inlined.length;
		sPhoto.data.inlined.mime_type = NULL;
		if ( g_file_get_contents( psPerson -> pnNewImage, ( gchar ** ) ppnData, ( gsize * ) pnLength, NULL ) ) {
			e_contact_set( psContact, E_CONTACT_PHOTO, &sPhoto );
		}
	} else if ( psPerson -> psImage == NULL ) {
		e_contact_set( psContact, E_CONTACT_PHOTO, NULL );
	}
}

void addrRefreshButtonClicked(GtkWidget *, gpointer);

void writeCallback( GObject *psSource, GAsyncResult *pnRes, gpointer pUserData ) {
	EBookClient *psClient = E_BOOK_CLIENT( psSource );
	EBookQuery *psQuery;
	EContact *psContact;
	GError *psError = NULL;
	GList *psList;
	EContactAddress sPrivateAddress;
	EContactAddress *psPrivateAddress;
	EContactAddress sBusinessAddress;
	EContactAddress *psBusinessAddress;

	Debug( KERN_DEBUG, "called\n" );

	psQuery = e_book_query_any_field_contains( "" );
	if ( psQuery == NULL ) {
		Debug( KERN_DEBUG, "Couldn't create query.\n" );
		g_object_unref( psClient );
		return;
	}

	Debug( KERN_DEBUG, "psPersonsList: %p\n", psPersonsList );
	for ( psList = psPersonsList; psList != NULL; psList = psList -> next ) {
		struct sPerson *psPerson = psList -> data;
		bool bSet = false;

		if ( psPerson != NULL && psPerson -> nFlags != PERSON_FLAG_UNCHANGED ) {
			Debug( KERN_DEBUG, "Get contact....\n" );
			if( e_book_client_get_contact_sync( psClient, psPerson -> pnId, &psContact, NULL, &psError ) == FALSE ) {
				Debug( KERN_DEBUG, "Got contact....\n" );
				if ( psPerson -> nFlags & PERSON_FLAG_NEW ) {
					Debug( KERN_DEBUG, "Add new person!\n" );
					EContact *psNewContact = e_contact_new();

					if ( psNewContact != NULL ) {
						e_contact_set( psNewContact, E_CONTACT_GIVEN_NAME, psPerson -> pnFirstName );
						e_contact_set( psNewContact, E_CONTACT_FAMILY_NAME, psPerson -> pnLastName );
						e_contact_set( psNewContact, E_CONTACT_FULL_NAME, psPerson -> pnDisplayName );
						e_contact_set( psNewContact, E_CONTACT_ORG, psPerson -> pnCompany );
						e_contact_set( psNewContact, E_CONTACT_PHONE_BUSINESS, psPerson -> pnBusinessPhone );
						e_contact_set( psNewContact, E_CONTACT_PHONE_HOME, psPerson -> pnPrivatePhone );
						e_contact_set( psNewContact, E_CONTACT_PHONE_BUSINESS_FAX, psPerson -> pnBusinessFax );
						e_contact_set( psNewContact, E_CONTACT_PHONE_HOME_FAX, psPerson -> pnPrivateFax );
						e_contact_set( psNewContact, E_CONTACT_PHONE_MOBILE, psPerson -> pnPrivateMobile );
						e_contact_set( psNewContact, E_CONTACT_TITLE, psPerson -> pnTitle );

						psPrivateAddress = &sPrivateAddress;
						memset( psPrivateAddress, 0, sizeof( EContactAddress ) );

						if ( psPerson -> pnPrivateStreet != NULL && strlen( psPerson -> pnPrivateStreet ) > 0 ) {
							psPrivateAddress -> street = psPerson -> pnPrivateStreet;
							bSet = TRUE;
						}
						if ( psPerson -> pnPrivateCity != NULL && strlen( psPerson -> pnPrivateCity ) > 0 ) {
							psPrivateAddress -> locality = psPerson -> pnPrivateCity;
							bSet = TRUE;
						}
						if ( psPerson -> pnPrivateZipCode != NULL && strlen( psPerson -> pnPrivateZipCode ) > 0 ) {
							psPrivateAddress -> code = psPerson -> pnPrivateZipCode;
							bSet = TRUE;
						}
						if ( psPerson -> pnPrivateCountry != NULL && strlen( psPerson -> pnPrivateCountry ) > 0 ) {
							psPrivateAddress -> country = psPerson -> pnPrivateCountry;
							bSet = TRUE;
						}

						if ( bSet == TRUE ) {
							Debug( KERN_DEBUG, "Set contact private address\n" );
							e_contact_set( psNewContact, E_CONTACT_ADDRESS_HOME, psPrivateAddress );
						}

						bSet = FALSE;

						psBusinessAddress = &sBusinessAddress;
						memset( psBusinessAddress, 0, sizeof( EContactAddress ) );

						if ( psPerson -> pnBusinessStreet != NULL && strlen( psPerson -> pnBusinessStreet ) > 0 ) {
							psBusinessAddress -> street = psPerson -> pnBusinessStreet;
							bSet = TRUE;
						}
						if ( psPerson -> pnBusinessCity != NULL && strlen( psPerson -> pnBusinessCity ) > 0 ) {
							psBusinessAddress -> locality = psPerson -> pnBusinessCity;
							bSet = TRUE;
						}
						if ( psPerson -> pnBusinessZipCode != NULL && strlen( psPerson -> pnBusinessZipCode ) > 0 ) {
							psBusinessAddress -> code = psPerson -> pnBusinessZipCode;
							bSet = TRUE;
						}
						if ( psPerson -> pnBusinessCountry != NULL && strlen( psPerson -> pnBusinessCountry ) > 0 ) {
							psBusinessAddress -> country = psPerson -> pnBusinessCountry;
							bSet = TRUE;
						}

						if ( bSet == TRUE ) {
							Debug( KERN_DEBUG, "Set contact business address\n" );
							e_contact_set( psNewContact, E_CONTACT_ADDRESS_WORK, psBusinessAddress );
						}

						ebookSetImage( psNewContact, psPerson );
						e_book_client_add_contact_sync( psClient, psNewContact, NULL, NULL, &psError );
						Debug( KERN_DEBUG, "Contact added '%s'\n", psPerson -> pnDisplayName );

						bSet = FALSE;
					}
					psPerson -> nFlags = PERSON_FLAG_UNCHANGED;
				}
			} else {
				if ( psPerson -> nFlags & PERSON_FLAG_CHANGED ) {
					const char *pnWork = e_contact_get_const( psContact, E_CONTACT_PHONE_BUSINESS );
					const char *pnHome = e_contact_get_const( psContact, E_CONTACT_PHONE_HOME );
					const char *pnFax = e_contact_get_const( psContact, E_CONTACT_PHONE_HOME_FAX );
					const char *pnBusinessFax = e_contact_get_const( psContact, E_CONTACT_PHONE_BUSINESS_FAX );
					const char *pnMobile = e_contact_get_const( psContact, E_CONTACT_PHONE_MOBILE );
					const char *pnFirstName = e_contact_get_const( psContact, E_CONTACT_GIVEN_NAME );
					const char *pnLastName = e_contact_get_const( psContact, E_CONTACT_FAMILY_NAME );
					const char *pnCompany = e_contact_get_const( psContact, E_CONTACT_ORG );
					const char *pnTitle = e_contact_get_const( psContact, E_CONTACT_TITLE );

					Debug( KERN_DEBUG, "Contact changed '%s'\n", psPerson -> pnDisplayName );

					if ( pnCompany != NULL && strcmp( psPerson -> pnCompany, pnCompany ) ) {
						e_contact_set( psContact, E_CONTACT_ORG, psPerson -> pnCompany );
					}
					if ( pnTitle != NULL && strcmp( psPerson -> pnTitle, pnTitle ) ) {
						e_contact_set( psContact, E_CONTACT_TITLE, psPerson -> pnTitle );
					}

					if ( ( pnFirstName != NULL && strcmp( psPerson -> pnFirstName, pnFirstName ) ) || ( pnLastName != NULL && strcmp( psPerson -> pnLastName, pnLastName ) ) ) {
						gchar *pnTmp = g_strdup_printf( "%s %s", psPerson -> pnFirstName, psPerson -> pnLastName );
						e_contact_set( psContact, E_CONTACT_FULL_NAME, pnTmp );
						//g_free( pnTmp );

						if ( strcmp( psPerson -> pnFirstName, pnFirstName ) ) {
							e_contact_set( psContact, E_CONTACT_GIVEN_NAME, psPerson -> pnFirstName );
						}
						if ( strcmp( psPerson -> pnLastName, pnLastName ) ) {
							e_contact_set( psContact, E_CONTACT_FAMILY_NAME, psPerson -> pnLastName );
						}
					}

					if (
						( pnWork == NULL && psPerson -> pnBusinessPhone != NULL ) ||
						( pnWork != NULL && psPerson -> pnBusinessPhone == NULL ) ||
						strcmp( psPerson -> pnBusinessPhone, pnWork ) ) {
						e_contact_set( psContact, E_CONTACT_PHONE_BUSINESS, psPerson -> pnBusinessPhone );
					}
					if (
						( pnHome == NULL && psPerson -> pnPrivatePhone != NULL ) ||
						( pnHome != NULL && psPerson -> pnPrivatePhone == NULL ) ||
						strcmp( psPerson -> pnPrivatePhone, pnHome ) ) {
						e_contact_set( psContact, E_CONTACT_PHONE_HOME, psPerson -> pnPrivatePhone );
					}
					if (
						( pnFax == NULL && psPerson -> pnPrivateFax != NULL ) ||
						( pnFax != NULL && psPerson -> pnPrivateFax == NULL ) ||
						strcmp( psPerson -> pnPrivateFax, pnFax ) ) {
						e_contact_set( psContact, E_CONTACT_PHONE_HOME_FAX, psPerson -> pnPrivateFax );
					}
					if (
						( pnBusinessFax == NULL && psPerson -> pnBusinessFax != NULL ) ||
						( pnBusinessFax != NULL && psPerson -> pnBusinessFax == NULL ) ||
						strcmp( psPerson -> pnPrivateFax, pnBusinessFax ) ) {
						e_contact_set( psContact, E_CONTACT_PHONE_BUSINESS_FAX, psPerson -> pnBusinessFax );
					}
					if (
						( pnMobile == NULL && psPerson -> pnPrivateMobile != NULL ) ||
						( pnMobile != NULL && psPerson -> pnPrivateMobile == NULL ) ||
						strcmp( psPerson -> pnPrivateMobile, pnMobile ) ) {
						e_contact_set( psContact, E_CONTACT_PHONE_MOBILE, psPerson -> pnPrivateMobile );
					}

					psPrivateAddress = e_contact_get( psContact, E_CONTACT_ADDRESS_HOME );
					if ( psPrivateAddress == NULL ) {
						psPrivateAddress = &sPrivateAddress;
						memset( psPrivateAddress, 0, sizeof( EContactAddress ) );
					}

					if ( psPerson -> pnPrivateStreet != NULL && strlen( psPerson -> pnPrivateStreet ) > 0 ) {
						psPrivateAddress -> street = psPerson -> pnPrivateStreet;
						bSet = TRUE;
					}
					if ( psPerson -> pnPrivateCity != NULL && strlen( psPerson -> pnPrivateCity ) > 0 ) {
						psPrivateAddress -> locality = psPerson -> pnPrivateCity;
						bSet = TRUE;
					}
					if ( psPerson -> pnPrivateZipCode != NULL && strlen( psPerson -> pnPrivateZipCode ) > 0 ) {
						psPrivateAddress -> code = psPerson -> pnPrivateZipCode;
						bSet = TRUE;
					}
					if ( psPerson -> pnPrivateCountry != NULL && strlen( psPerson -> pnPrivateCountry ) > 0 ) {
						psPrivateAddress -> country = psPerson -> pnPrivateCountry;
						bSet = TRUE;
					}

					if ( bSet == TRUE ) {
						Debug( KERN_DEBUG, "Set contact private address\n" );
						e_contact_set( psContact, E_CONTACT_ADDRESS_HOME, psPrivateAddress );
					}

					bSet = FALSE;

					psBusinessAddress = e_contact_get( psContact, E_CONTACT_ADDRESS_WORK );
					if ( psBusinessAddress == NULL ) {
						psBusinessAddress = &sBusinessAddress;
						memset( psBusinessAddress, 0, sizeof( EContactAddress ) );
					}

					if ( psPerson -> pnBusinessStreet != NULL && strlen( psPerson -> pnBusinessStreet ) > 0 ) {
						psBusinessAddress -> street = psPerson -> pnBusinessStreet;
						bSet = TRUE;
					}
					if ( psPerson -> pnBusinessCity != NULL && strlen( psPerson -> pnBusinessCity ) > 0 ) {
						psBusinessAddress -> locality = psPerson -> pnBusinessCity;
						bSet = TRUE;
					}
					if ( psPerson -> pnBusinessZipCode != NULL && strlen( psPerson -> pnBusinessZipCode ) > 0 ) {
						psBusinessAddress -> code = psPerson -> pnBusinessZipCode;
						bSet = TRUE;
					}
					if ( psPerson -> pnBusinessCountry != NULL && strlen( psPerson -> pnBusinessCountry ) > 0 ) {
						psBusinessAddress -> country = psPerson -> pnBusinessCountry;
						bSet = TRUE;
					}

					if ( bSet == TRUE ) {
						Debug( KERN_DEBUG, "Set contact business address\n" );
						e_contact_set( psContact, E_CONTACT_ADDRESS_WORK, psBusinessAddress );
					}

					ebookSetImage( psContact, psPerson );
					e_book_client_modify_contact_sync( psClient, psContact, NULL, &psError );
				}

				if ( psPerson -> nFlags & PERSON_FLAG_DELETED ) {
					e_book_client_remove_contact_by_uid_sync( psClient, psPerson -> pnId, NULL, &psError );
					Debug( KERN_DEBUG, "Contact deleted '%s'\n", psPerson -> pnDisplayName );
				}

				psPerson -> nFlags = PERSON_FLAG_UNCHANGED;
			}
		}
	}

	e_book_query_unref( psQuery );
	g_object_unref( psClient );

	addrRefreshButtonClicked(NULL, GINT_TO_POINTER(1));
}

/**
 * \brief Save ebook list
 * \return error code
 */
static int ebookSaveBook( void ) {
	GError *psError = NULL;
	EBookClient *psClient;
	const gchar *pnUri = NULL;

	pnUri = ebookGetSelectedBook();
	if ( pnUri == NULL || strlen( pnUri ) == 0 ) {
		psClient = e_book_client_new_system( &psError );
	} else {
		psClient = e_book_client_new_from_uri( pnUri, &psError );
	}

	Debug( KERN_DEBUG, "e_client_open\n" );
	e_client_open( E_CLIENT( psClient ), TRUE, NULL, writeCallback, NULL );

	return 0;
}

/**
 * \brief Display ebook preferences window
 */
static void ebookPreferences( void ) {
	GtkWidget *psDialog;
	GtkWidget *psComboBox;
	GtkWidget *psLabel;
	GtkWidget *psBox;
	GError *psError;
	ESourceList *psAddressbooks;
	GSList *psGroups, *psGroup;
	int nIndex = 0;

	if ( !e_book_client_get_sources( &psAddressbooks, &psError ) ) {
		Debug( KERN_WARNING, "Could not retrieve addressbooks, abort!\n" );
		return;
	}
	
	psGroups = e_source_list_peek_groups( psAddressbooks );
	if ( psGroups == NULL ) {
		Debug( KERN_WARNING, "No groups!\n" );
		return;
	}

	psDialog = gtk_dialog_new_with_buttons( _( "Evolution Preferences" ), NULL, 0, GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE, NULL );
	psComboBox = gtk_combo_box_new_text();
	psLabel = gtk_label_new( "" );

	gtk_label_set_markup( GTK_LABEL( psLabel ), _( "<b>Select addressbook:</b>" ) );
	psBox = gtk_dialog_get_content_area( GTK_DIALOG( psDialog ) );
	gtk_box_pack_start( GTK_BOX( psBox ), psLabel, FALSE, TRUE, 10 );

	for ( psGroup = psGroups; psGroup != NULL; psGroup = psGroup -> next ) {
		GSList *psSources, *psSource;

		psSources = e_source_group_peek_sources( psGroup -> data );

		for ( psSource = psSources; psSource != NULL; psSource = psSource -> next ) {
			ESource *psESource = E_SOURCE( psSource -> data );

			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), e_source_peek_name( psESource ) );

			if ( ebookGetSelectedBook() != NULL ) {
				if ( !strcmp( e_source_get_uri( psESource ), ebookGetSelectedBook() ) ) {
					gtk_combo_box_set_active( GTK_COMBO_BOX( psComboBox ), nIndex );
				}
			} else {
				gtk_combo_box_set_active( GTK_COMBO_BOX( psComboBox ), 0 );
			}

			nIndex++;
		}
	}
	
	gtk_box_pack_start( GTK_BOX( psBox ), psComboBox, FALSE, TRUE, 5 );
	gtk_widget_set_size_request( psDialog, 300, 120 );
	gtk_widget_show( GTK_WIDGET( psLabel ) );
	gtk_widget_show( GTK_WIDGET( psComboBox ) );
	gtk_dialog_run( GTK_DIALOG( psDialog ) );

	prefsAddNone( getActiveProfile(), "/plugins/evolution" );

	nIndex = 0;
	for ( psGroup = psGroups; psGroup != NULL; psGroup = psGroup -> next ) {
		GSList *psSources, *psSource;

		psSources = e_source_group_peek_sources( psGroup -> data );

		for ( psSource = psSources; psSource != NULL; psSource = psSource -> next ) {
			ESource *psESource = E_SOURCE( psSource -> data );

			if ( gtk_combo_box_get_active( GTK_COMBO_BOX( psComboBox ) ) == nIndex ) {
				ebookSetSelectedBook( e_source_get_uri( psESource ) );
			}

			nIndex++;
		}
	}

	gtk_widget_destroy( psDialog );
	SavePreferences( getActiveProfile() );
	freePersons();
	ebookReadBook();
}

#else

/**
 * \brief Read ebook list
 * \return error code
 */
static int ebookReadBook( void ) {
	EBook *psBook;
	EBookQuery *psQuery;
	EContact *psContact;
	EContactPhoto *psPhoto;
	GdkPixbufLoader *psLoader;
	GList *psContacts;
	GList *psList;
	GError *psError = NULL;
	EContactAddress *psPrivateAddress = NULL;
	EContactAddress *psBusinessAddress = NULL;
	const gchar *pnUri = NULL;

	psQuery = e_book_query_any_field_contains( "" );
	if ( psQuery == NULL ) {
		Debug( KERN_WARNING, "Couldn't create query.\n" );
		return -1;
	}

	pnUri = ebookGetSelectedBook();
	if ( pnUri == NULL || strlen( pnUri ) == 0 ) {
		psBook = e_book_new_system_addressbook( &psError );
	} else {
		psBook = e_book_new_from_uri( pnUri, &psError );
	}

	if ( !e_book_open( psBook, TRUE, NULL ) ) {
		Debug( KERN_WARNING, "Couldn't load system addressbook.\n" );
		e_book_query_unref( psQuery );
		g_object_unref( psBook );
		return -2;
	}

	if ( !e_book_get_contacts( psBook, psQuery, &psContacts, NULL ) ) {
		Debug( KERN_WARNING, "Couldn't get query results.\n" );
		e_book_query_unref( psQuery );
		g_object_unref( psBook );
		return -3;
	}

	e_book_query_unref( psQuery );
	g_object_unref( psBook );

	if ( psContacts == NULL ) {
		Debug( KERN_WARNING, "No contacts found.\n" );
		return -4;
	}

	for ( psList = psContacts; psList != NULL; psList = psList -> next ) {
		GHashTable *psTable = NULL;
		const gchar *pnDisplayName;

		g_return_val_if_fail( E_IS_CONTACT( psList -> data ), -5 );
		psContact = E_CONTACT( psList -> data );

		pnDisplayName = e_contact_get_const( psContact, E_CONTACT_FULL_NAME );

		if ( pnDisplayName == NULL ) {
			continue;
		}

		psTable = g_hash_table_new( NULL, NULL );

		AddInfo( psTable, PERSON_ID, e_contact_get_const( psContact, E_CONTACT_UID ) );
		AddInfo( psTable, PERSON_FIRST_NAME, e_contact_get_const( psContact, E_CONTACT_GIVEN_NAME ) );
		AddInfo( psTable, PERSON_LAST_NAME, e_contact_get_const( psContact, E_CONTACT_FAMILY_NAME ) );
		AddInfo( psTable, PERSON_DISPLAY_NAME, pnDisplayName );
		AddInfo( psTable, PERSON_COMPANY, e_contact_get_const( psContact, E_CONTACT_ORG ) );
		AddInfo( psTable, PERSON_TITLE, e_contact_get_const( psContact, E_CONTACT_TITLE ) );
		AddInfo( psTable, PERSON_BUSINESS_PHONE, e_contact_get_const( psContact, E_CONTACT_PHONE_BUSINESS ) );
		AddInfo( psTable, PERSON_BUSINESS_FAX, e_contact_get_const( psContact, E_CONTACT_PHONE_BUSINESS_FAX ) );
		AddInfo( psTable, PERSON_PRIVATE_PHONE, e_contact_get_const( psContact, E_CONTACT_PHONE_HOME ) );
		AddInfo( psTable, PERSON_PRIVATE_FAX, e_contact_get_const( psContact, E_CONTACT_PHONE_HOME_FAX ) );
		AddInfo( psTable, PERSON_PRIVATE_MOBILE, e_contact_get_const( psContact, E_CONTACT_PHONE_MOBILE ) );
		psPrivateAddress = e_contact_get( psContact, E_CONTACT_ADDRESS_HOME );
		if ( psPrivateAddress != NULL ) {
			AddInfo( psTable, PERSON_PRIVATE_STREET, psPrivateAddress -> street );
			AddInfo( psTable, PERSON_PRIVATE_CITY, psPrivateAddress -> locality );
			AddInfo( psTable, PERSON_PRIVATE_ZIPCODE, psPrivateAddress -> code );
			AddInfo( psTable, PERSON_PRIVATE_COUNTRY, psPrivateAddress -> country );
		}
		psBusinessAddress = e_contact_get( psContact, E_CONTACT_ADDRESS_WORK );
		if ( psBusinessAddress != NULL ) {
			AddInfo( psTable, PERSON_BUSINESS_STREET, psBusinessAddress -> street );
			AddInfo( psTable, PERSON_BUSINESS_CITY, psBusinessAddress -> locality );
			AddInfo( psTable, PERSON_BUSINESS_ZIPCODE, psBusinessAddress -> code );
			AddInfo( psTable, PERSON_BUSINESS_COUNTRY, psBusinessAddress -> country );
		}

		psPhoto = e_contact_get( psContact, E_CONTACT_PHOTO );
		if ( psPhoto != NULL ) {
			psLoader = gdk_pixbuf_loader_new();
			if ( gdk_pixbuf_loader_write( psLoader, psPhoto -> data.inlined.data, psPhoto -> data.inlined.length, NULL ) ) {
				AddInfo( psTable, PERSON_IMAGE, ( const gchar * ) gdk_pixbuf_loader_get_pixbuf( psLoader ) );
			}
			gdk_pixbuf_loader_close( psLoader, NULL );
			e_contact_photo_free( psPhoto );
		}

		AddPerson( psTable, FALSE );
		if ( psPrivateAddress != NULL ) {
			e_contact_address_free( psPrivateAddress );
		}
		if ( psBusinessAddress != NULL ) {
			e_contact_address_free( psBusinessAddress );
		}
		g_hash_table_destroy( psTable );
	}

	g_list_free( psContacts );

	return 0;
}

static void ebookSetImage( EContact *psContact, struct sPerson *psPerson ) {
	if ( psPerson -> pnNewImage != NULL ) {
		EContactPhoto sPhoto;
		guchar **ppnData;
		gsize *pnLength;

		sPhoto.type = E_CONTACT_PHOTO_TYPE_INLINED;
		ppnData = &sPhoto.data.inlined.data;
		pnLength = ( gsize * ) &sPhoto.data.inlined.length;
		sPhoto.data.inlined.mime_type = NULL;
		if ( g_file_get_contents( psPerson -> pnNewImage, ( gchar ** ) ppnData, ( gsize * ) pnLength, NULL ) ) {
			e_contact_set( psContact, E_CONTACT_PHOTO, &sPhoto );
		}
	} else if ( psPerson -> psImage == NULL ) {
		e_contact_set( psContact, E_CONTACT_PHOTO, NULL );
	}
}

/**
 * \brief Save ebook list
 * \return error code
 */
static int ebookSaveBook( void ) {
	EBook *psBook;
	EBookQuery *psQuery;
	EContact *psContact;
	GError *psError = NULL;
	GList *psList;
	EContactAddress sPrivateAddress;
	EContactAddress *psPrivateAddress;
	EContactAddress sBusinessAddress;
	EContactAddress *psBusinessAddress;
	const gchar *pnUri;

	pnUri = ebookGetSelectedBook();
	if ( pnUri == NULL || strlen( pnUri ) == 0 ) {
		psBook = e_book_new_system_addressbook( &psError );
	} else {
		psBook = e_book_new_from_uri( pnUri, &psError );
	}

	if ( !e_book_open( psBook, TRUE, NULL ) ) {
		Debug( KERN_DEBUG, "Couldn't load system addressbook.\n" );
		return -1;
	}

	psQuery = e_book_query_any_field_contains( "" );
	if ( psQuery == NULL ) {
		Debug( KERN_DEBUG, "Couldn't create query.\n" );
		g_object_unref( psBook );
		return -2;
	}

	for ( psList = psPersonsList; psList != NULL; psList = psList -> next ) {
		struct sPerson *psPerson = psList -> data;
		bool bSet = false;

		if ( psPerson != NULL && psPerson -> nFlags != PERSON_FLAG_UNCHANGED ) {
			if( e_book_get_contact( psBook, psPerson -> pnId, &psContact, NULL ) == FALSE ) {
				if ( psPerson -> nFlags & PERSON_FLAG_NEW ) {
					Debug( KERN_DEBUG, "Add new person!\n" );
					EContact *psNewContact = e_contact_new();

					if ( psNewContact != NULL ) {
						e_contact_set( psNewContact, E_CONTACT_GIVEN_NAME, psPerson -> pnFirstName );
						e_contact_set( psNewContact, E_CONTACT_FAMILY_NAME, psPerson -> pnLastName );
						e_contact_set( psNewContact, E_CONTACT_FULL_NAME, psPerson -> pnDisplayName );
						e_contact_set( psNewContact, E_CONTACT_ORG, psPerson -> pnCompany );
						e_contact_set( psNewContact, E_CONTACT_PHONE_BUSINESS, psPerson -> pnBusinessPhone );
						e_contact_set( psNewContact, E_CONTACT_PHONE_HOME, psPerson -> pnPrivatePhone );
						e_contact_set( psNewContact, E_CONTACT_PHONE_BUSINESS_FAX, psPerson -> pnBusinessFax );
						e_contact_set( psNewContact, E_CONTACT_PHONE_HOME_FAX, psPerson -> pnPrivateFax );
						e_contact_set( psNewContact, E_CONTACT_PHONE_MOBILE, psPerson -> pnPrivateMobile );
						e_contact_set( psNewContact, E_CONTACT_TITLE, psPerson -> pnTitle );

						psPrivateAddress = &sPrivateAddress;
						memset( psPrivateAddress, 0, sizeof( EContactAddress ) );

						if ( psPerson -> pnPrivateStreet != NULL && strlen( psPerson -> pnPrivateStreet ) > 0 ) {
							psPrivateAddress -> street = psPerson -> pnPrivateStreet;
							bSet = TRUE;
						}
						if ( psPerson -> pnPrivateCity != NULL && strlen( psPerson -> pnPrivateCity ) > 0 ) {
							psPrivateAddress -> locality = psPerson -> pnPrivateCity;
							bSet = TRUE;
						}
						if ( psPerson -> pnPrivateZipCode != NULL && strlen( psPerson -> pnPrivateZipCode ) > 0 ) {
							psPrivateAddress -> code = psPerson -> pnPrivateZipCode;
							bSet = TRUE;
						}
						if ( psPerson -> pnPrivateCountry != NULL && strlen( psPerson -> pnPrivateCountry ) > 0 ) {
							psPrivateAddress -> country = psPerson -> pnPrivateCountry;
							bSet = TRUE;
						}

						if ( bSet == TRUE ) {
							Debug( KERN_DEBUG, "Set contact private address\n" );
							e_contact_set( psNewContact, E_CONTACT_ADDRESS_HOME, psPrivateAddress );
						}

						bSet = FALSE;

						psBusinessAddress = &sBusinessAddress;
						memset( psBusinessAddress, 0, sizeof( EContactAddress ) );

						if ( psPerson -> pnBusinessStreet != NULL && strlen( psPerson -> pnBusinessStreet ) > 0 ) {
							psBusinessAddress -> street = psPerson -> pnBusinessStreet;
							bSet = TRUE;
						}
						if ( psPerson -> pnBusinessCity != NULL && strlen( psPerson -> pnBusinessCity ) > 0 ) {
							psBusinessAddress -> locality = psPerson -> pnBusinessCity;
							bSet = TRUE;
						}
						if ( psPerson -> pnBusinessZipCode != NULL && strlen( psPerson -> pnBusinessZipCode ) > 0 ) {
							psBusinessAddress -> code = psPerson -> pnBusinessZipCode;
							bSet = TRUE;
						}
						if ( psPerson -> pnBusinessCountry != NULL && strlen( psPerson -> pnBusinessCountry ) > 0 ) {
							psBusinessAddress -> country = psPerson -> pnBusinessCountry;
							bSet = TRUE;
						}

						if ( bSet == TRUE ) {
							Debug( KERN_DEBUG, "Set contact business address\n" );
							e_contact_set( psNewContact, E_CONTACT_ADDRESS_WORK, psBusinessAddress );
						}

						ebookSetImage( psNewContact, psPerson );
						e_book_add_contact( psBook, psNewContact, NULL );
						Debug( KERN_DEBUG, "Contact added '%s'\n", psPerson -> pnDisplayName );

						bSet = FALSE;
					}
					psPerson -> nFlags = PERSON_FLAG_UNCHANGED;
				}
			} else {
				if ( psPerson -> nFlags & PERSON_FLAG_CHANGED ) {
					const char *pnWork = e_contact_get_const( psContact, E_CONTACT_PHONE_BUSINESS );
					const char *pnHome = e_contact_get_const( psContact, E_CONTACT_PHONE_HOME );
					const char *pnFax = e_contact_get_const( psContact, E_CONTACT_PHONE_HOME_FAX );
					const char *pnBusinessFax = e_contact_get_const( psContact, E_CONTACT_PHONE_BUSINESS_FAX );
					const char *pnMobile = e_contact_get_const( psContact, E_CONTACT_PHONE_MOBILE );
					const char *pnFirstName = e_contact_get_const( psContact, E_CONTACT_GIVEN_NAME );
					const char *pnLastName = e_contact_get_const( psContact, E_CONTACT_FAMILY_NAME );
					const char *pnCompany = e_contact_get_const( psContact, E_CONTACT_ORG );
					const char *pnTitle = e_contact_get_const( psContact, E_CONTACT_TITLE );

					Debug( KERN_DEBUG, "Contact changed '%s'\n", psPerson -> pnDisplayName );

					if ( pnCompany != NULL && strcmp( psPerson -> pnCompany, pnCompany ) ) {
						e_contact_set( psContact, E_CONTACT_ORG, psPerson -> pnCompany );
					}
					if ( pnTitle != NULL && strcmp( psPerson -> pnTitle, pnTitle ) ) {
						e_contact_set( psContact, E_CONTACT_TITLE, psPerson -> pnTitle );
					}

					if ( ( pnFirstName != NULL && strcmp( psPerson -> pnFirstName, pnFirstName ) ) || ( pnLastName != NULL && strcmp( psPerson -> pnLastName, pnLastName ) ) ) {
						gchar *pnTmp = g_strdup_printf( "%s %s", psPerson -> pnFirstName, psPerson -> pnLastName );
						e_contact_set( psContact, E_CONTACT_FULL_NAME, pnTmp );
						//g_free( pnTmp );

						if ( strcmp( psPerson -> pnFirstName, pnFirstName ) ) {
							e_contact_set( psContact, E_CONTACT_GIVEN_NAME, psPerson -> pnFirstName );
						}
						if ( strcmp( psPerson -> pnLastName, pnLastName ) ) {
							e_contact_set( psContact, E_CONTACT_FAMILY_NAME, psPerson -> pnLastName );
						}
					}

					if (
						( pnWork == NULL && psPerson -> pnBusinessPhone != NULL ) ||
						( pnWork != NULL && psPerson -> pnBusinessPhone == NULL ) ||
						strcmp( psPerson -> pnBusinessPhone, pnWork ) ) {
						e_contact_set( psContact, E_CONTACT_PHONE_BUSINESS, psPerson -> pnBusinessPhone );
					}
					if (
						( pnHome == NULL && psPerson -> pnPrivatePhone != NULL ) ||
						( pnHome != NULL && psPerson -> pnPrivatePhone == NULL ) ||
						strcmp( psPerson -> pnPrivatePhone, pnHome ) ) {
						e_contact_set( psContact, E_CONTACT_PHONE_HOME, psPerson -> pnPrivatePhone );
					}
					if (
						( pnFax == NULL && psPerson -> pnPrivateFax != NULL ) ||
						( pnFax != NULL && psPerson -> pnPrivateFax == NULL ) ||
						strcmp( psPerson -> pnPrivateFax, pnFax ) ) {
						e_contact_set( psContact, E_CONTACT_PHONE_HOME_FAX, psPerson -> pnPrivateFax );
					}
					if (
						( pnBusinessFax == NULL && psPerson -> pnBusinessFax != NULL ) ||
						( pnBusinessFax != NULL && psPerson -> pnBusinessFax == NULL ) ||
						strcmp( psPerson -> pnPrivateFax, pnBusinessFax ) ) {
						e_contact_set( psContact, E_CONTACT_PHONE_BUSINESS_FAX, psPerson -> pnBusinessFax );
					}
					if (
						( pnMobile == NULL && psPerson -> pnPrivateMobile != NULL ) ||
						( pnMobile != NULL && psPerson -> pnPrivateMobile == NULL ) ||
						strcmp( psPerson -> pnPrivateMobile, pnMobile ) ) {
						e_contact_set( psContact, E_CONTACT_PHONE_MOBILE, psPerson -> pnPrivateMobile );
					}

					psPrivateAddress = e_contact_get( psContact, E_CONTACT_ADDRESS_HOME );
					if ( psPrivateAddress == NULL ) {
						psPrivateAddress = &sPrivateAddress;
						memset( psPrivateAddress, 0, sizeof( EContactAddress ) );
					}

					if ( psPerson -> pnPrivateStreet != NULL && strlen( psPerson -> pnPrivateStreet ) > 0 ) {
						psPrivateAddress -> street = psPerson -> pnPrivateStreet;
						bSet = TRUE;
					}
					if ( psPerson -> pnPrivateCity != NULL && strlen( psPerson -> pnPrivateCity ) > 0 ) {
						psPrivateAddress -> locality = psPerson -> pnPrivateCity;
						bSet = TRUE;
					}
					if ( psPerson -> pnPrivateZipCode != NULL && strlen( psPerson -> pnPrivateZipCode ) > 0 ) {
						psPrivateAddress -> code = psPerson -> pnPrivateZipCode;
						bSet = TRUE;
					}
					if ( psPerson -> pnPrivateCountry != NULL && strlen( psPerson -> pnPrivateCountry ) > 0 ) {
						psPrivateAddress -> country = psPerson -> pnPrivateCountry;
						bSet = TRUE;
					}

					if ( bSet == TRUE ) {
						Debug( KERN_DEBUG, "Set contact private address\n" );
						e_contact_set( psContact, E_CONTACT_ADDRESS_HOME, psPrivateAddress );
					}

					bSet = FALSE;

					psBusinessAddress = e_contact_get( psContact, E_CONTACT_ADDRESS_WORK );
					if ( psBusinessAddress == NULL ) {
						psBusinessAddress = &sBusinessAddress;
						memset( psBusinessAddress, 0, sizeof( EContactAddress ) );
					}

					if ( psPerson -> pnBusinessStreet != NULL && strlen( psPerson -> pnBusinessStreet ) > 0 ) {
						psBusinessAddress -> street = psPerson -> pnBusinessStreet;
						bSet = TRUE;
					}
					if ( psPerson -> pnBusinessCity != NULL && strlen( psPerson -> pnBusinessCity ) > 0 ) {
						psBusinessAddress -> locality = psPerson -> pnBusinessCity;
						bSet = TRUE;
					}
					if ( psPerson -> pnBusinessZipCode != NULL && strlen( psPerson -> pnBusinessZipCode ) > 0 ) {
						psBusinessAddress -> code = psPerson -> pnBusinessZipCode;
						bSet = TRUE;
					}
					if ( psPerson -> pnBusinessCountry != NULL && strlen( psPerson -> pnBusinessCountry ) > 0 ) {
						psBusinessAddress -> country = psPerson -> pnBusinessCountry;
						bSet = TRUE;
					}

					if ( bSet == TRUE ) {
						Debug( KERN_DEBUG, "Set contact business address\n" );
						e_contact_set( psContact, E_CONTACT_ADDRESS_WORK, psBusinessAddress );
					}

					ebookSetImage( psContact, psPerson );
					e_book_commit_contact( psBook, psContact, NULL );
				}

				if ( psPerson -> nFlags & PERSON_FLAG_DELETED ) {
					e_book_remove_contact( psBook, psPerson -> pnId, NULL );
					Debug( KERN_DEBUG, "Contact deleted '%s'\n", psPerson -> pnDisplayName );
				}

				psPerson -> nFlags = PERSON_FLAG_UNCHANGED;
			}
		}
	}

	e_book_query_unref( psQuery );
	g_object_unref( psBook );

	return 0;
}

/**
 * \brief Display ebook preferences window
 */
static void ebookPreferences( void ) {
	GtkWidget *psDialog;
	GtkWidget *psComboBox;
	GtkWidget *psLabel;
	GtkWidget *psBox;
	GError *psError;
	ESourceList *psAddressbooks;
	GSList *psGroups, *psGroup;
	int nIndex = 0;

	if ( !e_book_get_addressbooks( &psAddressbooks, &psError ) ) {
		Debug( KERN_WARNING, "Could not retrieve addressbooks, abort!\n" );
		return;
	}
	
	psGroups = e_source_list_peek_groups( psAddressbooks );
	if ( psGroups == NULL ) {
		Debug( KERN_WARNING, "No groups!\n" );
		return;
	}

	psDialog = gtk_dialog_new_with_buttons( _( "Evolution Preferences" ), NULL, 0, GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE, NULL );
	psComboBox = gtk_combo_box_new_text();
	psLabel = gtk_label_new( "" );

	gtk_label_set_markup( GTK_LABEL( psLabel ), _( "<b>Select addressbook:</b>" ) );
	psBox = gtk_dialog_get_content_area( GTK_DIALOG( psDialog ) );
	gtk_box_pack_start( GTK_BOX( psBox ), psLabel, FALSE, TRUE, 10 );

	for ( psGroup = psGroups; psGroup != NULL; psGroup = psGroup -> next ) {
		GSList *psSources, *psSource;

		psSources = e_source_group_peek_sources( psGroup -> data );

		for ( psSource = psSources; psSource != NULL; psSource = psSource -> next ) {
			ESource *psESource = E_SOURCE( psSource -> data );

			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), e_source_peek_name( psESource ) );

			if ( ebookGetSelectedBook() != NULL ) {
				if ( !strcmp( e_source_get_uri( psESource ), ebookGetSelectedBook() ) ) {
					gtk_combo_box_set_active( GTK_COMBO_BOX( psComboBox ), nIndex );
				}
			} else {
				gtk_combo_box_set_active( GTK_COMBO_BOX( psComboBox ), 0 );
			}

			nIndex++;
		}
	}
	
	gtk_box_pack_start( GTK_BOX( psBox ), psComboBox, FALSE, TRUE, 5 );
	gtk_widget_set_size_request( psDialog, 300, 120 );
	gtk_widget_show( GTK_WIDGET( psLabel ) );
	gtk_widget_show( GTK_WIDGET( psComboBox ) );
	gtk_dialog_run( GTK_DIALOG( psDialog ) );

	prefsAddNone( getActiveProfile(), "/plugins/evolution" );

	nIndex = 0;
	for ( psGroup = psGroups; psGroup != NULL; psGroup = psGroup -> next ) {
		GSList *psSources, *psSource;

		psSources = e_source_group_peek_sources( psGroup -> data );

		for ( psSource = psSources; psSource != NULL; psSource = psSource -> next ) {
			ESource *psESource = E_SOURCE( psSource -> data );

			if ( gtk_combo_box_get_active( GTK_COMBO_BOX( psComboBox ) ) == nIndex ) {
				ebookSetSelectedBook( e_source_get_uri( psESource ) );
			}

			nIndex++;
		}
	}

	gtk_widget_destroy( psDialog );
	SavePreferences( getActiveProfile() );
	freePersons();
	ebookReadBook();
}
#endif

/** book definition */
static struct sAddressBook sEbook = {
	ebookReadBook,
	ebookSaveBook,
	/* Read flags */
	PERSON_FIRST_NAME | PERSON_LAST_NAME | PERSON_DISPLAY_NAME | PERSON_COMPANY |
	PERSON_TITLE | PERSON_IMAGE	| PERSON_PRIVATE_PHONE | PERSON_PRIVATE_MOBILE |
	PERSON_PRIVATE_FAX | PERSON_PRIVATE_STREET | PERSON_PRIVATE_CITY |
	PERSON_PRIVATE_ZIPCODE | PERSON_PRIVATE_COUNTRY | PERSON_BUSINESS_PHONE |
	PERSON_BUSINESS_FAX | PERSON_BUSINESS_STREET | PERSON_BUSINESS_CITY |
	PERSON_BUSINESS_ZIPCODE | PERSON_BUSINESS_COUNTRY,
	/* Write flags */
	PERSON_FIRST_NAME | PERSON_LAST_NAME | PERSON_DISPLAY_NAME | PERSON_COMPANY |
	PERSON_TITLE | PERSON_IMAGE | PERSON_PRIVATE_PHONE | PERSON_PRIVATE_MOBILE |
	PERSON_PRIVATE_FAX | PERSON_PRIVATE_STREET | PERSON_PRIVATE_CITY |
	PERSON_PRIVATE_ZIPCODE | PERSON_PRIVATE_COUNTRY | PERSON_BUSINESS_PHONE |
	PERSON_BUSINESS_FAX | PERSON_BUSINESS_STREET | PERSON_BUSINESS_CITY |
	PERSON_BUSINESS_ZIPCODE | PERSON_BUSINESS_COUNTRY,
	ebookPreferences
};

MODULE_INIT( PLUGIN_TYPE_BOOK, _( "Evolution Addressbook" ), &sEbook, NULL, NULL );
