% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/reactives.R
\name{observeEvent}
\alias{observeEvent}
\alias{eventReactive}
\title{Event handler}
\usage{
observeEvent(
  eventExpr,
  handlerExpr,
  event.env = parent.frame(),
  event.quoted = FALSE,
  handler.env = parent.frame(),
  handler.quoted = FALSE,
  ...,
  label = NULL,
  suspended = FALSE,
  priority = 0,
  domain = getDefaultReactiveDomain(),
  autoDestroy = TRUE,
  ignoreNULL = TRUE,
  ignoreInit = FALSE,
  once = FALSE
)

eventReactive(
  eventExpr,
  valueExpr,
  event.env = parent.frame(),
  event.quoted = FALSE,
  value.env = parent.frame(),
  value.quoted = FALSE,
  ...,
  label = NULL,
  domain = getDefaultReactiveDomain(),
  ignoreNULL = TRUE,
  ignoreInit = FALSE
)
}
\arguments{
\item{eventExpr}{A (quoted or unquoted) expression that represents the event;
this can be a simple reactive value like \code{input$click}, a call to a
reactive expression like \code{dataset()}, or even a complex expression
inside curly braces}

\item{handlerExpr}{The expression to call whenever \code{eventExpr} is
invalidated. This should be a side-effect-producing action (the return
value will be ignored). It will be executed within an \code{\link[=isolate]{isolate()}}
scope.}

\item{event.env}{The parent environment for the reactive expression. By default,
this is the calling environment, the same as when defining an ordinary
non-reactive expression. If \code{eventExpr} is a quosure and \code{event.quoted} is \code{TRUE},
then \code{event.env} is ignored.}

\item{event.quoted}{If it is \code{TRUE}, then the \code{\link[=quote]{quote()}}ed value of \code{eventExpr}
will be used when \code{eventExpr} is evaluated. If \code{eventExpr} is a quosure and you
would like to use its expression as a value for \code{eventExpr}, then you must set
\code{event.quoted} to \code{TRUE}.}

\item{handler.env}{The parent environment for the reactive expression. By default,
this is the calling environment, the same as when defining an ordinary
non-reactive expression. If \code{handlerExpr} is a quosure and \code{handler.quoted} is \code{TRUE},
then \code{handler.env} is ignored.}

\item{handler.quoted}{If it is \code{TRUE}, then the \code{\link[=quote]{quote()}}ed value of \code{handlerExpr}
will be used when \code{handlerExpr} is evaluated. If \code{handlerExpr} is a quosure and you
would like to use its expression as a value for \code{handlerExpr}, then you must set
\code{handler.quoted} to \code{TRUE}.}

\item{...}{Currently not used.}

\item{label}{A label for the observer or reactive, useful for debugging.}

\item{suspended}{If \code{TRUE}, start the observer in a suspended state. If
\code{FALSE} (the default), start in a non-suspended state.}

\item{priority}{An integer or numeric that controls the priority with which
this observer should be executed. An observer with a given priority level
will always execute sooner than all observers with a lower priority level.
Positive, negative, and zero values are allowed.}

\item{domain}{See \link{domains}.}

\item{autoDestroy}{If \code{TRUE} (the default), the observer will be
automatically destroyed when its domain (if any) ends.}

\item{ignoreNULL}{Whether the action should be triggered (or value
calculated, in the case of \code{eventReactive}) when the input is
\code{NULL}. See Details.}

\item{ignoreInit}{If \code{TRUE}, then, when this \code{observeEvent} is
first created/initialized, ignore the \code{handlerExpr} (the second
argument), whether it is otherwise supposed to run or not. The default is
\code{FALSE}. See Details.}

\item{once}{Whether this \code{observeEvent} should be immediately destroyed
after the first time that the code in \code{handlerExpr} is run. This
pattern is useful when you want to subscribe to a event that should only
happen once.}

\item{valueExpr}{The expression that produces the return value of the
\code{eventReactive}. It will be executed within an \code{\link[=isolate]{isolate()}}
scope.}

\item{value.env}{The parent environment for the reactive expression. By default,
this is the calling environment, the same as when defining an ordinary
non-reactive expression. If \code{valueExpr} is a quosure and \code{value.quoted} is \code{TRUE},
then \code{value.env} is ignored.}

\item{value.quoted}{If it is \code{TRUE}, then the \code{\link[=quote]{quote()}}ed value of \code{valueExpr}
will be used when \code{valueExpr} is evaluated. If \code{valueExpr} is a quosure and you
would like to use its expression as a value for \code{valueExpr}, then you must set
\code{value.quoted} to \code{TRUE}.}
}
\value{
\code{observeEvent} returns an observer reference class object (see
\code{\link[=observe]{observe()}}). \code{eventReactive} returns a reactive expression
object (see \code{\link[=reactive]{reactive()}}).
}
\description{
Respond to "event-like" reactive inputs, values, and expressions. As of Shiny
1.6.0, we recommend using \code{\link[=bindEvent]{bindEvent()}} instead of \code{eventReactive()} and
\code{observeEvent()}. This is because \code{bindEvent()} can be composed with
\code{\link[=bindCache]{bindCache()}}, and because it can also be used with \code{render} functions (like
\code{\link[=renderText]{renderText()}} and \code{\link[=renderPlot]{renderPlot()}}).
}
\details{
Shiny's reactive programming framework is primarily designed for calculated
values (reactive expressions) and side-effect-causing actions (observers)
that respond to \emph{any} of their inputs changing. That's often what is
desired in Shiny apps, but not always: sometimes you want to wait for a
specific action to be taken from the user, like clicking an
\code{\link[=actionButton]{actionButton()}}, before calculating an expression or taking an
action. A reactive value or expression that is used to trigger other
calculations in this way is called an \emph{event}.

These situations demand a more imperative, "event handling" style of
programming that is possible--but not particularly intuitive--using the
reactive programming primitives \code{\link[=observe]{observe()}} and
\code{\link[=isolate]{isolate()}}. \code{observeEvent} and \code{eventReactive} provide
straightforward APIs for event handling that wrap \code{observe} and
\code{isolate}.

Use \code{observeEvent} whenever you want to \emph{perform an action} in
response to an event. (Note that "recalculate a value" does not generally
count as performing an action--see \code{eventReactive} for that.) The first
argument is the event you want to respond to, and the second argument is a
function that should be called whenever the event occurs. Note that
\code{observeEvent()} is equivalent to using \code{observe() \%>\% bindEvent()} and as of
Shiny 1.6.0, we recommend the latter.

Use \code{eventReactive} to create a \emph{calculated value} that only
updates in response to an event. This is just like a normal
\link[=reactive]{reactive expression} except it ignores all the usual
invalidations that come from its reactive dependencies; it only invalidates
in response to the given event. Note that
\code{eventReactive()} is equivalent to using \code{reactive() \%>\% bindEvent()} and as of
Shiny 1.6.0, we recommend the latter.
}
\section{ignoreNULL and ignoreInit}{


Both \code{observeEvent} and \code{eventReactive} take an \code{ignoreNULL}
parameter that affects behavior when the \code{eventExpr} evaluates to
\code{NULL} (or in the special case of an \code{\link[=actionButton]{actionButton()}},
\code{0}). In these cases, if \code{ignoreNULL} is \code{TRUE}, then an
\code{observeEvent} will not execute and an \code{eventReactive} will raise a
silent \link[=validate]{validation} error. This is useful behavior if you
don't want to do the action or calculation when your app first starts, but
wait for the user to initiate the action first (like a "Submit" button);
whereas \code{ignoreNULL=FALSE} is desirable if you want to initially perform
the action/calculation and just let the user re-initiate it (like a
"Recalculate" button).

Likewise, both \code{observeEvent} and \code{eventReactive} also take in an
\code{ignoreInit} argument. By default, both of these will run right when they
are created (except if, at that moment, \code{eventExpr} evaluates to \code{NULL}
and \code{ignoreNULL} is \code{TRUE}). But when responding to a click of an action
button, it may often be useful to set \code{ignoreInit} to \code{TRUE}. For
example, if you're setting up an \code{observeEvent} for a dynamically created
button, then \code{ignoreInit = TRUE} will guarantee that the action (in
\code{handlerExpr}) will only be triggered when the button is actually clicked,
instead of also being triggered when it is created/initialized. Similarly,
if you're setting up an \code{eventReactive} that responds to a dynamically
created button used to refresh some data (then returned by that \code{eventReactive}),
then you should use \verb{eventReactive([...], ignoreInit = TRUE)} if you want
to let the user decide if/when they want to refresh the data (since, depending
on the app, this may be a computationally expensive operation).

Even though \code{ignoreNULL} and \code{ignoreInit} can be used for similar
purposes they are independent from one another. Here's the result of combining
these:

\describe{
\item{\code{ignoreNULL = TRUE} and \code{ignoreInit = FALSE}}{
This is the default. This combination means that \code{handlerExpr}/
\code{valueExpr} will run every time that \code{eventExpr} is not
\code{NULL}. If, at the time of the creation of the
\code{observeEvent}/\code{eventReactive}, \code{eventExpr} happens
to \emph{not} be \code{NULL}, then the code runs.
}
\item{\code{ignoreNULL = FALSE} and \code{ignoreInit = FALSE}}{
This combination means that \code{handlerExpr}/\code{valueExpr} will
run every time no matter what.
}
\item{\code{ignoreNULL = FALSE} and \code{ignoreInit = TRUE}}{
This combination means that \code{handlerExpr}/\code{valueExpr} will
\emph{not} run when the \code{observeEvent}/\code{eventReactive} is
created (because \code{ignoreInit = TRUE}), but it will run every
other time.
}
\item{\code{ignoreNULL = TRUE} and \code{ignoreInit = TRUE}}{
This combination means that \code{handlerExpr}/\code{valueExpr} will
\emph{not} run when the \code{observeEvent}/\code{eventReactive} is
created (because  \code{ignoreInit = TRUE}). After that,
\code{handlerExpr}/\code{valueExpr} will run every time that
\code{eventExpr} is not \code{NULL}.
}
}
}

\examples{
## Only run examples in interactive R sessions
if (interactive()) {

  ## App 1: Sample usage
  shinyApp(
    ui = fluidPage(
      column(4,
        numericInput("x", "Value", 5),
        br(),
        actionButton("button", "Show")
      ),
      column(8, tableOutput("table"))
    ),
    server = function(input, output) {
      # Take an action every time button is pressed;
      # here, we just print a message to the console
      observeEvent(input$button, {
        cat("Showing", input$x, "rows\n")
      })
      # The observeEvent() above is equivalent to:
      # observe({
      #    cat("Showing", input$x, "rows\n")
      #   }) \%>\%
      #   bindEvent(input$button)

      # Take a reactive dependency on input$button, but
      # not on any of the stuff inside the function
      df <- eventReactive(input$button, {
        head(cars, input$x)
      })
      output$table <- renderTable({
        df()
      })
    }
  )

  ## App 2: Using `once`
  shinyApp(
    ui = basicPage( actionButton("go", "Go")),
    server = function(input, output, session) {
      observeEvent(input$go, {
        print(paste("This will only be printed once; all",
              "subsequent button clicks won't do anything"))
      }, once = TRUE)
      # The observeEvent() above is equivalent to:
      # observe({
      #   print(paste("This will only be printed once; all",
      #         "subsequent button clicks won't do anything"))
      #   }) \%>\%
      #   bindEvent(input$go, once = TRUE)
    }
  )

  ## App 3: Using `ignoreInit` and `once`
  shinyApp(
    ui = basicPage(actionButton("go", "Go")),
    server = function(input, output, session) {
      observeEvent(input$go, {
        insertUI("#go", "afterEnd",
                 actionButton("dynamic", "click to remove"))

        # set up an observer that depends on the dynamic
        # input, so that it doesn't run when the input is
        # created, and only runs once after that (since
        # the side effect is remove the input from the DOM)
        observeEvent(input$dynamic, {
          removeUI("#dynamic")
        }, ignoreInit = TRUE, once = TRUE)
      })
    }
  )
}
}
\seealso{
\code{\link[=actionButton]{actionButton()}}
}
