% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot-methods.R
\name{tree_layout}
\alias{tree_layout}
\title{Returns a data table defining the line segments of a phylogenetic tree.}
\usage{
tree_layout(phy, ladderize = FALSE)
}
\arguments{
\item{phy}{(Required). The \code{\link{phylo}} or \code{\link{phyloseq-class}}
object (which must contain a \code{\link{phylo}}genetic tree)
that you want to converted to \code{\link{data.table}}s
suitable for plotting with \code{\link[ggplot2]{ggplot}}2.}

\item{ladderize}{(Optional). Boolean or character string (either
\code{FALSE}, \code{TRUE}, or \code{"left"}).
Default is \code{FALSE} (no ladderization).
This parameter specifies whether or not to \code{\link[ape]{ladderize}} the tree 
(i.e., reorder nodes according to the depth of their enclosed
subtrees) prior to plotting.
This tends to make trees more aesthetically pleasing and legible in
a graphical display.
When \code{TRUE} or \code{"right"}, ``right'' ladderization is used.
When set to \code{FALSE}, no ladderization is applied.
When set to \code{"left"}, the reverse direction
(``left'' ladderization) is applied.}
}
\value{
A list of two \code{\link{data.table}}s, containing respectively 
 a \code{data.table} of edge segment coordinates, named \code{edgeDT},
 and a \code{data.table} of vertical connecting segments, named \code{vertDT}.
 See \code{example} below for a simple demonstration.
}
\description{
This function takes a \code{\link{phylo}} or \code{\link{phyloseq-class}} object
and returns a list of two \code{\link{data.table}}s suitable for plotting
a phylogenetic tree with \code{\link[ggplot2]{ggplot}}2.
}
\examples{
library("ggplot2")
data("esophagus")
phy = phy_tree(esophagus)
phy <- ape::root(phy, "65_2_5", resolve.root=TRUE)
treeSegs0 = tree_layout(phy)
treeSegs1 = tree_layout(esophagus)
edgeMap = aes(x=xleft, xend=xright, y=y, yend=y)
vertMap = aes(x=x, xend=x, y=vmin, yend=vmax)
p0 = ggplot(treeSegs0$edgeDT, edgeMap) + geom_segment() + geom_segment(vertMap, data=treeSegs0$vertDT)
p1 = ggplot(treeSegs1$edgeDT, edgeMap) + geom_segment() + geom_segment(vertMap, data=treeSegs1$vertDT)
print(p0)
print(p1)
plot_tree(esophagus, "treeonly")
plot_tree(esophagus, "treeonly", ladderize="left")
}
\seealso{
An early example of this functionality was borrowed directly, with permission,
from the package called \code{ggphylo}, 
released on GitHub at:
\url{https://github.com/gjuggler/ggphylo}
by its author Gregory Jordan \email{gjuggler@gmail.com}.
That original phyloseq internal function, \code{tree.layout}, has been
completely replaced by this smaller and much faster user-accessible 
function that utilizes performance enhancements from standard 
\code{\link{data.table}} magic as well as \code{\link{ape-package}}
internal C code.
}
