// Copyright Earl Warren <contact@earl-warren.org>
// Copyright Loïc Dachary <loic@dachary.org>
// SPDX-License-Identifier: MIT

package generic

import (
	"context"
	"testing"

	"code.forgejo.org/f3/gof3/v3/path"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestTreeInit(t *testing.T) {
	tree := NewTree(newTestOptions())
	assert.NotNil(t, tree)

	assert.True(t, tree == tree.GetSelf())
	tree.SetSelf(tree)
	assert.True(t, tree == tree.GetSelf())

	other := NewTree(newTestOptions())
	assert.False(t, other == tree.GetSelf())
	assert.False(t, other.GetSelf() == tree)
	assert.False(t, other.GetSelf() == tree.GetSelf())
}

func TestTreeRoot(t *testing.T) {
	tree := NewTree(newTestOptions())
	root := NewNode()
	tree.SetRoot(root)
	assert.True(t, root == tree.GetRoot())
}

func TestTreePageSize(t *testing.T) {
	tree := NewTree(newTestOptions())
	assert.EqualValues(t, DefaultPageSize, tree.GetPageSize())
	pageSize := int(100)
	tree.GetDriver().SetPageSize(pageSize)
	assert.EqualValues(t, pageSize, tree.GetPageSize())
}

func TestTreeAllocateID(t *testing.T) {
	tree := NewTree(newTestOptions())
	assert.True(t, tree.AllocateID())
}

func TestTreeFactoryDerived(t *testing.T) {
	tree := newTestTree()
	root := tree.Factory(context.Background(), kindTestNodeLevelOne)
	tree.SetRoot(root)
	assert.True(t, root == tree.GetRoot())
	assert.True(t, kindTestNodeLevelOne == root.GetKind())
	assert.True(t, tree.GetSelf() == root.GetTree().GetSelf())

	leveltwo := tree.Factory(context.Background(), kindTestNodeLevelTwo)
	leveltwo.SetParent(root)
	assert.True(t, root == leveltwo.GetParent())
	assert.True(t, kindTestNodeLevelTwo == leveltwo.GetKind())
	assert.True(t, tree.GetSelf() == leveltwo.GetTree().GetSelf())
}

func TestTreeApply(t *testing.T) {
	tree := newTestTree()
	root := tree.Factory(context.Background(), kindTestNodeLevelOne)
	tree.SetRoot(root)

	assert.True(t, tree.Apply(context.Background(), path.NewPath(), nil))
	assert.True(t, tree.Apply(context.Background(), path.NewPathFromString(NewElementNode, "/"),
		NewApplyOptions(func(ctx context.Context, parent, path path.Path, node NodeInterface) {
			require.Equal(t, root, node)
		})),
	)
}

func TestTreeApplyAndGet(t *testing.T) {
	tree := newTestTree()
	root := tree.Factory(context.Background(), kindTestNodeLevelOne)
	tree.SetRoot(root)

	assert.True(t, tree.ApplyAndGet(context.Background(), path.NewPath(), nil))
	assert.True(t, tree.ApplyAndGet(context.Background(), path.NewPathFromString(NewElementNode, "/"),
		NewApplyOptions(func(ctx context.Context, parent, path path.Path, node NodeInterface) {
			require.Equal(t, root, node)
		})),
	)
}
