/*
 * Show information about all nodes (or empty positions) in the torus
 */
#include "basil_mysql.h"
#include <err.h>

int main(int ac, char **av)
{
	const char query[] =	"SELECT processor_id, "
				"	processor_type, "
				"	processor_status, "
				"	alloc_mode "
				"FROM  processor "
				"WHERE x_coord = ? "
				"AND   y_coord = ? "
				"AND   z_coord = ? ";
	enum query_par {
			PAR_X,
			PAR_Y,
			PAR_Z,
			PARAM_COUNT	/* sentinel */
	};
	MYSQL_BIND	bind_parm[PARAM_COUNT];
	int		par[PARAM_COUNT];

	enum query_columns {
			/* integer data */
			COL_NID,
			/* string data */
			COL_TYPE,
			COL_STATE,
			COL_ALLOC,
#define			CDATA_TYPE	(COL_TYPE  - COL_NID)
#define			CDATA_STATE	(COL_STATE - COL_NID)
#define			CDATA_ALLOC	(COL_ALLOC - COL_NID)
#define			CDATA_COUNT	(CDATA_ALLOC + 1)
			COLUMN_COUNT	/* sentinel */
	};
	MYSQL_BIND	bind_cols[COLUMN_COUNT];
	unsigned int	node_id;
	char 		c_data[CDATA_COUNT][BASIL_STRING_SIZE];
	my_bool		is_null[COLUMN_COUNT];
	my_bool		error[COLUMN_COUNT];

	MYSQL		*handle;
	MYSQL_STMT	*stmt;

	int		rc = -1, i, x_max, y_max, z_max, is_gemini;

	memset(bind_parm, 0, sizeof(bind_parm));
	for (i = 0; i < PARAM_COUNT; i ++) {
		bind_parm[i].buffer_type = MYSQL_TYPE_LONG;
		bind_parm[i].buffer	 = (char *)&par[i];
	}

	memset(bind_cols, 0, sizeof(bind_cols));
	for (i = 0; i < COLUMN_COUNT; i ++) {
		bind_cols[i].is_null = &is_null[i];
		bind_cols[i].error   = &error[i];
	}
	bind_cols[COL_NID].buffer_type	= MYSQL_TYPE_LONG;
	bind_parm[COL_NID].is_unsigned	= true;
	bind_cols[COL_NID].buffer	= (char *)&node_id;

	for (i = CDATA_TYPE; i < CDATA_COUNT; i++) {
		bind_cols[i].buffer_type   = MYSQL_TYPE_STRING;
		bind_cols[i].buffer	   = c_data[i];
		bind_cols[i].buffer_length = BASIL_STRING_SIZE;
	}

	handle = cray_connect_sdb();
	if (handle == NULL)
		errx(1, "can not connect to XTAdmin database on the SDB");

	is_gemini = cray_is_gemini_system(handle);
	if (is_gemini < 0)
		errx(1, "can not determine type of interconnect (SeaStar/Gemini)");
	else if (is_gemini)
		errx(1, "not meant to be used on Gemini/XE (torus positions are ambiguous)");

	if (!cray_get_torus_dimensions(handle, &x_max, &y_max, &z_max))
		errx(1, "can not determine torus dimensions");

	stmt = prepare_stmt(handle, query, bind_parm, PARAM_COUNT,
					   bind_cols, COLUMN_COUNT);
	if (stmt == NULL)
		goto err;

	for (par[PAR_X] = PAR_X; par[PAR_X] <= x_max; par[PAR_X]++) {
		for (par[PAR_Y] = 0; par[PAR_Y] <= y_max; par[PAR_Y]++) {
			for (par[PAR_Z] = 0; par[PAR_Z] <= z_max; par[PAR_Z]++) {

				if (exec_stmt(stmt, query, bind_cols,
							COLUMN_COUNT) < 0)
					goto err_close;

				if (!par[PAR_X] && !par[PAR_Y] && !par[PAR_Z])
					printf("  X   Y   Z    NID         TYPE"
							"    STATUS        MODE\n");

				printf("%3d %3d %3d", par[0], par[1], par[2]);

				if (mysql_stmt_fetch(stmt) == 0) {
					printf(" %6u   %10s %9s %11s", node_id,
						c_data[CDATA_TYPE],
						c_data[CDATA_STATE],
						c_data[CDATA_ALLOC]);
					if (strcmp(c_data[CDATA_TYPE], "compute") != 0 ||
					    strcmp(c_data[CDATA_ALLOC], "batch") != 0)
						printf("\t=> NOT USABLE");
					else if (strcmp(c_data[CDATA_STATE], "up") != 0)
						printf("\t=> DOWN");
				} else {
					/* Hole in the NID space (service node) */
					printf("      ?");
					printf("%35s\t=> HOLE", "");
				}
				printf("\n");
			}
		}
	}

	rc = 0;
err_close:
	if (mysql_stmt_close(stmt))
		warnx("error closing statement: %s",
			mysql_stmt_error(stmt));
err:
	mysql_close(handle);

	return rc ? EXIT_FAILURE : EXIT_SUCCESS;
}
